#include <mail/akita/http_api/include/server/context.h>
#include <mail/akita/http_api/include/reflection/error.h>
#include <butil/split_key_value.h>
#include <boost/lexical_cast.hpp>
#include <butil/http/extract_auth_domain.h>
#include <boost/range/algorithm/transform.hpp>
#include <yamail/data/serialization/json_writer.h>
#include <butil/http/extract_auth_domain.h>
#include <butil/network/real_ip_from_headers.h>
#include <mail/akita/service/include/auth_checker/auth_checker.h>


namespace akita {
namespace server {

Context::Context(const ymod_webserver::http::stream_ptr& stream, const http_getter::ClientModule& httpModule, boost::optional<u_int32_t> source)
    : request(stream->request())
    , logger(getLogger(getOptionalHeader("X-Request-Id").get_value_or("")))
    , method(stream->request()->url.make_full_path())
    , response_(stream)
    , httpPtr(
        httpModule.create(
            *stream->request(),
            http_getter::withLog(
                  httpModule.httpLogger(getOptionalArg("uid").get_value_or(""), requestId())
            )
        )
    )
    , tvm2source(std::move(source))
{
    parseCookieHeader();
}

void Context::response(const std::string& body, bool json, ymod_webserver::codes::code code) {
    response_->set_code(code);

    if (json) {
        response_->set_content_type("application", "json");
    } else {
        response_->set_content_type("text", "plain");
    }

    if (json) {
        std::string yandexuid;
        getCookie("yandexuid", yandexuid);

        LOGDOG_(logger, notice,
                log::where_name=method,
                log::yandexuid=yandexuid);
    }

    response_->result_body(body);
}

void Context::response(Reason error, const std::string& message, ymod_webserver::codes::code respCode) {
    const int code = static_cast<int>(error);
    const std::string reason = toString(error);
    std::string yandexuid;
    getCookie("yandexuid", yandexuid);

    LOGDOG_(logger, error,
            log::where_name=method,
            log::code=code,
            log::reason=reason,
            log::message=message,
            log::source=tvm2source,
            log::yandexuid=yandexuid);

    response_->set_code(respCode);
    response_->set_content_type("application", "json");
    response_->result_body(yamail::data::serialization::toJson(reflection::Error{code, message, reason}, "error").str());
}

std::set<std::string> Context::getArgsAsSet(const std::string& name) const {
    std::set<std::string> ret;
    boost::copy(request->url.params.equal_range(name)
                | boost::adaptors::map_values
                , std::inserter(ret, ret.end()));

    return ret;
}

boost::optional<std::string> Context::getOptionalArg(const std::string& name) const {
    auto iter = request->url.params.find(name);
    return iter == request->url.params.end() ? boost::none : boost::make_optional(iter->second);
}

boost::optional<std::string> Context::getOptionalHeader(const std::string& name) const {
    const std::string n = boost::algorithm::to_lower_copy(name);
    auto iter = request->headers.find(n);

    return iter == request->headers.end() ? boost::none : boost::make_optional(iter->second);
}

void Context::parseCookieHeader() {
    const auto c = getOptionalHeader("cookie");
    if (c) {
        std::vector<std::string> cookiesList;
        boost::split(cookiesList, *c, boost::is_any_of(";"));
        boost::transform(cookiesList, std::inserter(cookies, cookies.end()),
                         boost::bind(splitKeyValueTrimmed, _1));
    }
}

std::string Context::authDomain() const {
    return http::extractAuthDomain(getOptionalHeader("X-Original-Host").get_value_or(""), ".yandex.ru");
}

bool Context::getCookie(const std::string &name, std::string &value) const {
    const auto cookie = cookies.find(name);
    if (cookie != cookies.end()) {
        value = cookie->second;
        return true;
    }
    return false;
}

bool Context::hasCookie(const std::string& name) const {
    return cookies.find(name) != cookies.end();
}

bool Context::getArgument(const std::string& name, std::string& value) const {
    const auto arg = getOptionalArg(name);

    if (arg && !arg->empty()) {
        value = *arg;
    }

    return static_cast<bool>(arg);
}

bool Context::hasArgument(const std::string& name) const {
    return !(getOptionalArg(name).get_value_or("")).empty();
}

bool Context::hasHeader(const std::string& name) const {
    return !(getOptionalHeader(name).get_value_or("")).empty();
}

bool Context::getHeader(const std::string& name, std::string& value) const {
    const auto header = getOptionalHeader(name);

    if (header && !header->empty()) {
        value = *header;
    }

    return static_cast<bool>(header);
}

std::set<std::string> Context::getSidsToCheck() const {
    return getArgsAsSet("sids_to_check");
}

std::set<std::string> Context::getAttributesToCheck() const {
    return getArgsAsSet("attributes_to_check");
}

std::string Context::requestId() const {
    return getOptionalHeader("X-Request-Id").get_value_or("");
}

std::string Context::getRealIp() const {
    return real_ip::getFromHeaders(getOptionalHeader("X-Forwarded-For"), getOptionalHeader("X-Real-Ip"),
                                   request->context->remote_address);
}

std::string Context::getRealPort() const {
    if (const auto h = getOptionalHeader("X-Real-Port"); h && !h->empty()) {
        return *h;
    } else {
        return std::to_string(request->context->remote_port);
    }
}

bool Context::deferredEmailListRequest() const {
    return getOptionalArg("emails").get_value_or("") != "yes";
}

}
}
