#include <mail/akita/http_api/include/server/handlers/auth.h>
#include <mail/akita/http_api/include/reflection/auth.h>
#include <mail/akita/http_api/include/reflection/error.h>
#include <mail/akita/http_api/include/reflection/user_is_frozen_error.h>

#include <mail/akita/service/include/auth.h>

#include <mail/akita/service/include/auth_checker/auth_checker.h>
#include <mail_errors/error_code.h>
#include <mail/akita/service/include/errors.h>
#include <map>


namespace akita {
namespace server {
namespace handlers {

Reason getErrorCode(AuthCode authCode) {
    static const std::map<AuthCode, Reason> codesMap({
                           {AuthCode::noPassword,      Reason::authNoPassword}
                         , {AuthCode::noAuth,          Reason::authNoAuth}
                         , {AuthCode::internalProblem, Reason::authInternalProblem}
                         , {AuthCode::noMailbox,       Reason::authNoMailbox}
                         , {AuthCode::valid,           Reason::ok}
                         , {AuthCode::userIsBlocked,   Reason::authUserBlocked}
                         , {AuthCode::unallowedOAuthScope, Reason::authUnallowedScope}
                         , {AuthCode::userIsFrozen,    Reason::authFrozen}
                         , {AuthCode::wrongGuard,      Reason::authBadSessGuard}});
    auto found = codesMap.find(authCode);
    return (found == codesMap.end()) ? Reason::unknown : found->second;
}

void Auth::execute(ContextPtr ctx) const {
    authorize(makeBlackBox(ctx->httpPtr, config_->endpoint, ctx->logger, config_->scopes),
              *ctx, config_->provideSessGuard, [=](mail_errors::error_code e, AuthResultData authData) {
        if (e.category() == server::getTransportCategory()) {
            ctx->response(Reason::authInternalProblem, e.what());
        } else {
            auto authCode = static_cast<AuthCode>(e.value());
            if (authCode == AuthCode::valid || authCode == AuthCode::needReset) {
                ctx->response(reflection::reflectAuth(std::move(authData.account),
                                                      (authCode == AuthCode::needReset), authData.age));
            } else if (AuthCode::userIsFrozen == authCode) {
                const std::string responseBody = reflection::reflectUserIsFrozenError(std::move(authData.account),
                                                                                      getErrorCode(authCode));
                ctx->response(responseBody);
            } else {
                const auto err = getErrorCode(authCode);
                ctx->response(err, toString(err));
            }
        }
    });
}

}
}
}
