#include <mail/akita/http_api/include/server/handlers/check_cookies.h>

#include <mail/akita/service/include/blackbox/parse.h>
#include <mail/akita/service/include/logger.h>
#include <mail/akita/service/include/reflection/sessionid_response.h>
#include <yamail/data/serialization/json_writer.h>

namespace akita::server::handlers {

using namespace http_getter::detail::operators;

void CheckCookies::execute(ContextPtr ctx) const {
    HttpArguments postArgs;

    std::string sessionId;
    ctx->getCookie("Session_id", sessionId);
    if (sessionId.empty()) {
        ctx->response(Reason::authNoAuth, "Cookie Session_id is required", ymod_webserver::codes::code::bad_request);
        return;
    }
    postArgs.add("sessionid", sessionId);

    auto builder = ctx->httpPtr->toPOST(config_->endpoint)
        .postArgs(postArgs)
        .getArgs(
            "userip"_arg=ctx->getRealIp(), "host"_arg=ctx->authDomain(),
            "attributes"_arg=akita::blackbox::sessionid::timeZoneAttribute(),
            "aliases"_arg="12", "method"_arg="sessionid",
            "multisession"_arg="1", "format"_arg="json",
            "get_login_id"_arg="yes"
        );

    const auto answered = std::make_shared<bool>(false);

    ctx->httpPtr->req(builder)->call(
        "blackbox", http_getter::withDefaultHttpWrap([answered, ctx] (yhttp::response response) {
            using namespace blackbox;
            try {
                boost::variant<sessionid::Error, sessionid::Response> parsed = sessionid::process(response.body);
                if (0 == parsed.which()) {
                    Reason error;
                    std::string description;
                    ymod_webserver::codes::code httpCode;
                    std::tie(error, description, httpCode) = boost::get<sessionid::Error>(parsed);
                    LOGDOG_WHERE_(ctx->logger, debug, log::reason=description, log::bb_response=response.body);
                    ctx->response(error, description, httpCode);
                } else {
                    const auto parsedBbResponse = boost::get<sessionid::Response>(parsed);
                    const auto resp = sessionid::CheckCookies(parsedBbResponse);
                    ctx->response(yamail::data::serialization::toJson(resp, "check_cookies").str());
                }
            } catch (const boost::coroutines::detail::forced_unwind&) {
                throw;
            } catch (const std::exception& e) {
                ctx->response(Reason::authInternalProblem, e.what(), ymod_webserver::codes::code::internal_server_error);
            } catch (...) {
                ctx->response(Reason::authInternalProblem, "Unknown exception", ymod_webserver::codes::code::internal_server_error);
            }

            *answered = true;
    }), [answered, ctx] (mail_errors::error_code) {
        if (!*answered) {
            ctx->response(Reason::authInternalProblem, "Try limit", ymod_webserver::codes::code::internal_server_error);
        }
    });
}

} // namespace akita::server::handlers
