// Copyright (c) 2017 Yandex LLC. All rights reserved.
// Use of this source code is governed by a MPL 2.0
// license that can be found in the LICENSE file.
// Author: Vladimir Skipor <skipor@yandex-team.ru>

package main

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"regexp"
	"time"

	"github.com/yandex/pandora/cli"
	phttp "github.com/yandex/pandora/components/phttp/import"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/register"
)

type Sample struct {
	URL              string
	ShootTimeSeconds float64
}

type GunConfig struct {
	Target string `validate:"required"` // Configuration will fail, without target defined
}

type Gun struct {
	// Configured on construction.
	client http.Client
	expire time.Time
	conf   GunConfig
	// Configured on Bind, before shooting
	aggr core.Aggregator // May be your custom Aggregator.
	core.GunDeps
}

func NewGun(conf GunConfig) *Gun {
	return &Gun{conf: conf}
}

func (g *Gun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	tr := &http.Transport{
		MaxIdleConns:       1,
		IdleConnTimeout:    600 * time.Second,
		DisableCompression: true,
	}
	g.client = http.Client{Transport: tr} //keep-alive shooting
	g.aggr = aggr
	g.GunDeps = deps
	g.expire = time.Now().AddDate(0, 0, 1)
	return nil
}

func (g *Gun) Shoot(ammo core.Ammo) {
	customAmmo := ammo.(*Ammo) // Shoot will panic on unexpected ammo type. Panic cancels shooting.
	g.shoot(customAmmo)
}

func (g *Gun) shoot(ammo *Ammo) {
	code := 0

	payload := g.genPayload(ammo)
	req := g.makeReq(ammo, payload)

	sample := netsample.Acquire(ammo.Tag)

	rs, err := g.client.Do(req)
	if err != nil {
		code = 0
		fmt.Println(err)
	} else {
		code = rs.StatusCode
		if code == 200 {
			respBody, _ := ioutil.ReadAll(rs.Body)
			re := regexp.MustCompile(payload.Assert)
			if re.FindString(string(respBody)) == "" || payload.Assert == "" {
				code = rs.StatusCode
			} else {
				code = 314
			}
		}
		_ = rs.Body.Close()
	}
	defer func() {
		sample.SetProtoCode(code)
		g.aggr.Report(sample)
	}()
}

func main() {
	//debug.SetGCPercent(-1)
	// Standard imports.
	fs := coreimport.GetFs()
	coreimport.Import(fs)
	// May not be imported, if you don't need http guns and etc.
	phttp.Import(fs)

	// Custom imports. Integrate your custom types into configuration system.
	coreimport.RegisterCustomJSONProvider("akita_provider", func() core.Ammo { return &Ammo{} })

	register.Gun("akita_gun", NewGun, func() GunConfig {
		return GunConfig{
			Target: "default target",
		}
	})
	//	register.Gun("my-custom/no-default", NewGun)

	cli.Run()
}
