#include <mail/akita/service/include/auth_checker/auth_checker.h>
#include <mail/akita/service/include/auth_checker/exceptions.h>
#include <mail/akita/service/include/errors.h>

namespace akita {

std::string AuthCategory::message(int v) const {
    switch(AuthCode(v)) {
        case AuthCode::notFinished:
            return "authentication is not finished yet";
        case AuthCode::valid:
            return "no error";
        case AuthCode::needReset:
            return "authenticated, have to update authcookie";
        case AuthCode::noAuth:
            return "authentication required";
        case AuthCode::noMailbox:
            return "authenticated, no mailbox";
        case AuthCode::internalProblem:
            return "internal error";
        case AuthCode::noPassword:
            return "no password";
        case AuthCode::userIsBlocked:
            return "the user is blocked";
        case AuthCode::unallowedOAuthScope:
            return "oauth scope is not allowed";
        case AuthCode::userIsFrozen:
            return "the user is inactive (frozen or archived)";
        case AuthCode::wrongGuard:
            return "bad sessguard authorization cookie";
    }
    return "unknown enum code";
}

const AuthCategory& getAuthCategory() {
    static AuthCategory instance;
    return instance;
}

mail_errors::error_code::base_type make_error_code(AuthCode e) {
    return mail_errors::error_code::base_type(static_cast<int>(e), getAuthCategory());
}

mail_errors::error_code make_error(AuthCode e, std::string what) {
    return mail_errors::error_code(e, std::move(what));
}

namespace {
bool containsAllowedOAuthScope(const BlackBox& bb, const BlackBox::SessionResponse& response) {
    const auto& scopes = response.oAuthInfo.scopes;
    return scopes.end() != std::find_if(scopes.begin(), scopes.end(), [&] (const std::string& scope) {
        return bb.isAllowedOAuthScope(scope);
    });
}

std::string webMailAuthHost() {
    return "mail";
}

bool validAndSucceeded(AuthCode code) {
    return code == AuthCode::valid || code == AuthCode::needReset || code == AuthCode::userIsFrozen;
}

}

CommonCheckRequest commonCheckRequest(const AuthContext& context) {
    return CommonCheckRequest {
        .sidsToCheck=context.getSidsToCheck(),
        .attributesToCheck=context.getAttributesToCheck(),
        .deferredEmailListRequest=context.deferredEmailListRequest(),
        .authDomain=context.authDomain(),
        .realIp=context.getRealIp(),
        .realPort=context.getRealPort()
    };
}

AuthRequestFn createOAuthRequest(const AuthContext& context) {
    OAuthRequest request;
    request.common = commonCheckRequest(context);
    request.token = context.getOAuthToken();

    return [r = std::move(request)] (const BlackBox& bb, BlackBox::OnCheck h) {
        bb.checkOAuthAsync(std::move(r), std::move(h));
    };
}

AuthRequestFn createSSLAuthRequest(const AuthContext& context, bool provideSessGuard) {
    SessionRequestSSL request;
    request.common = commonCheckRequest(context);
    request.currentUid = context.currentUid();
    if (context.getSessionIdCookie(request.sessionID)) {
        request.authHost = webMailAuthHost() + context.authDomain();
    } else {
        throw NoSessionId("createSSLAuthRequest(): no SessionID found!");
    }
    if (provideSessGuard) {
        context.getSessGuardCookie(request.sessGuard);
    }

    return [r = std::move(request)] (const BlackBox& bb, BlackBox::OnCheck h) {
        bb.checkSessionSSLAsync(std::move(r), std::move(h));
    };
}

AuthRequestFn buildAuthRequest(const AuthContext& context, bool provideSessGuard) {
    if(context.hasOAuthToken()) {
        return createOAuthRequest(context);
    } else if (context.hasSessionIdCookie()) {
        return createSSLAuthRequest(context, provideSessGuard);
    }

    throw UnknownAuthPattern("Unknown authorization pattern!");
}

AuthCode processResponse(const BlackBox& bb, const BlackBox::SessionResponse& response)  {
    using SessionResponse = BlackBox::SessionResponse;
    AuthCode result = AuthCode::noAuth;
    const SessionResponse::Status status( response.status );
    switch( status ) {
        case SessionResponse::Status_valid:
            result = AuthCode::valid;
            break;

        case SessionResponse::Status_needReset:
            result = AuthCode::needReset;
            break;

        case SessionResponse::Status_wrongGuard:
            return AuthCode::wrongGuard;

        case SessionResponse::Status_expired :
        case SessionResponse::Status_disabled :
        case SessionResponse::Status_invalid :
        case SessionResponse::Status_noAuth :
        default:
            return AuthCode::noAuth;
    }
    if (response.mailStatus != SubscriptionMailStatus::undefined){
        if (response.mailStatus == SubscriptionMailStatus::noMailUser) {
            return AuthCode::noMailbox;
        } else if (response.mailStatus == SubscriptionMailStatus::frozen) {
            return AuthCode::userIsFrozen;
        }
    } else if (response.serviceUserId.empty()) {
        return AuthCode::noMailbox;
    }
    if ( !response.mailLoginAllowed ) {
        result = AuthCode::userIsBlocked;
    } else if ( !response.isMailish && !response.hasPassword && !response.isSSO) {
        result = AuthCode::noPassword;
    } else if ( response.isOAuth && !containsAllowedOAuthScope(bb, response) ) {
        result = AuthCode::unallowedOAuthScope;
    }

    return result;
}

void checkAuthAsync(BlackBoxPtr bb, AuthRequestFn request, OnAuthCheck h) {
    request(*bb, [h = std::move(h), bb] (mail_errors::error_code e, const BlackBox::SessionResponse response,
                 BlackBox::FeaturedAccountPtr myResultData) {
        if(e) {
            h(e, AuthResultData());
        } else try {
            AuthCode result(processResponse(*bb, response));

            AuthResultData resultData;
            if( validAndSucceeded(result) ) {
                resultData.account = std::move(myResultData);
            }

            resultData.uid = response.uid;
            resultData.age = response.age;

            h(mail_errors::error_code(result), std::move(resultData));
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const std::exception&) {
            h(mail_errors::error_code(AuthCode::notFinished), AuthResultData());
        }
    });
}

}
