#include <mail/akita/service/include/blackbox/parse.h>

#include <mail/akita/service/include/reflection/sessionid_response.h>
#include <ymod_webserver/codes.h>
#include <yamail/data/deserialization/json_reader.h>

namespace akita {
namespace blackbox {
namespace sessionid {

bool statusIs(const boost::property_tree::ptree& tree, const StatusId& id) {
    const auto statusNode = tree.get_child_optional("status");
    if (!statusNode) {
        return false;
    }

    const auto status = statusNode->get_optional<int>("id");
    if (!status) {
        return false;
    }

    return *status == static_cast<int>(id);
}

bool checkUserStatus(const Response& resp) {
    if (!resp.status.isOk()) {
        return false;
    }
    const boost::optional<User>& defaultUser = resp.defaultUser();
    if (!defaultUser || !defaultUser->status.isOk()) {
        return false;
    }
    return true;
}

boost::variant<Error, Response> process(const std::string& json) {
    using ymod_webserver::codes::code::internal_server_error;
    using ymod_webserver::codes::code::ok;
    using namespace server;

    try {
        const auto ptree = yamail::data::deserialization::json::toPtree(json);
        if (ptree.get_child_optional("exception")) {
            return std::make_tuple(Reason::authInternalProblem, ptree.get_child("error").get_value<std::string>(), internal_server_error);
        } else if (statusIs(ptree, StatusId::EXPIRED)) {
            return std::make_tuple(Reason::authNoAuth, "expired", ok);
        } else if (statusIs(ptree, StatusId::INVALID)) {
            return std::make_tuple(Reason::authNoAuth, ptree.get_child("error").get_value<std::string>(), ok);
        }

        const auto data = yamail::data::deserialization::fromPtree<Response>(ptree);
        if (data.status.isOk() && checkUserStatus(data)) {
            return data;
        }
        return std::make_tuple(Reason::authNoAuth, toString(Reason::authNoAuth), ok);
    } catch (const boost::property_tree::ptree_error&) {
        return std::make_tuple(Reason::unknown, "unexpected response from blackbox", internal_server_error);
    } catch (const yamail::data::deserialization::json::JsonToPtreeException&) {
        return std::make_tuple(Reason::unknown, "unexpected response from blackbox", internal_server_error);
    } catch (const std::exception& e) {
        return std::make_tuple(Reason::unknown, e.what(), internal_server_error);
    }
}

} // namespace sessionid
} // namespace blackbox
} // namespace akita

