#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <mail/akita/service/include/blackbox/internal/blackbox_impl.h>
#include <mail/akita/service/include/blackbox/internal/dbfield_names.h>
#include <mail/akita/service/include/blackbox/internal/session_response_factory.h>
#include <boost/range/algorithm/find_if.hpp>

namespace akita
{

namespace blackbox
{

template<class Resp>
std::string getUserTicket(const Resp& resp) {
    try {
        return bb::UserTicket(&resp).get();
    } catch (...) { }

    return "";
}

bool SessionResponseFactory::checkAlias(const bb::SessionResp& response, bb::AliasList::Item::Type alias) {
    const bb::AliasList aliasList(&response);
    const auto& lst = aliasList.getAliases();
    return boost::find_if(lst, [alias](const auto& a) {
        return a.type() == alias;
    }) != lst.end();
}

bool SessionResponseFactory::isMailish(const bb::SessionResp& response) {
    return checkAlias(response, bb::AliasList::Item::Mailish);
}

bool SessionResponseFactory::isSSO(const bb::SessionResp& response) {
    return checkAlias(response, bb::AliasList::Item::SSO);
}

SubscriptionMailStatus SessionResponseFactory::getSubscriptionMailStatus(const bb::SessionResp & response) {
    const auto attrs = bb::Attributes(&response);
    if (!attrs.has(Attributes::subscriptionMailStatusAttr())) {
        return SubscriptionMailStatus::undefined;
    }
    if (attrs.get(Attributes::subscriptionMailStatusAttr()) == "1") {
        return SubscriptionMailStatus::active;
    }
    if (attrs.get(Attributes::subscriptionMailStatusAttr()) == "2") {
        return SubscriptionMailStatus::frozen;
    }
    return SubscriptionMailStatus::noMailUser;
}

SessionResponseFactory::SessionResponseFactory()
{ }

bool SessionResponseFactory::hasPassword( const bb::SessionResp & response  ) {
    const auto attrs = bb::Attributes(&response);
    return attrs.has(Attributes::havePasswordAttr()) && attrs.get(Attributes::havePasswordAttr()) == "1";
}

std::time_t SessionResponseFactory::age( const bb::SessionResp & response ) {
    bb::AuthInfo authInfo(&response);
    if( !authInfo.age().empty() ) {
        try {
            return boost::lexical_cast<std::time_t>(authInfo.age());
        } catch( const boost::bad_lexical_cast & ) {
            throw std::runtime_error("Bad authInfo.age(): " + authInfo.age() );
        }
    }
    return 0;
}

std::string SessionResponseFactory::getUid( const bb::SessionResp & response  ) {
    return bb::Uid(&response).uid();
}

bool SessionResponseFactory::mailLoginAllowed(const std::string &mailLoginRule)
{
    const std::string loginDenied( "0" );
    return mailLoginRule != loginDenied;
}

SessionResponseFactory::OAuthInfo SessionResponseFactory::getOAuthInfo( const bb::SessionResp& response ) {
    const bb::OAuthInfo oauth(&response);
    const auto& info = oauth.getInfo();
    const auto scope = info.find("scope");
    OAuthInfo result;

    if (scope != info.end()) {
        boost::algorithm::split(result.scopes, scope->second, boost::is_any_of(" "));
    }

    return result;
}

SessionResponseFactory&
SessionResponseFactory::fromValidCookieResp(const bb::MultiSessionResp& sessionInfo,
                                            const bb::SessionResp& accInfo) {
    bb::DBFields dbFields( &accInfo );

    _result.mailLoginAllowed = mailLoginAllowed(dbFields.get(DBFieldsNames::mailLoginRule()));
    _result.serviceUserId = dbFields.get(DBFieldsNames::mailServiceUserId());
    _result.status = getStatus(accInfo);
    _result.hasPassword = hasPassword(accInfo);
    _result.isMailish = isMailish(accInfo);
    _result.isSSO = isSSO(accInfo);
    _result.mailStatus = getSubscriptionMailStatus(accInfo);
    _result.uid = getUid(accInfo);
    _result.age = age(accInfo);
    _userTicket = getUserTicket(sessionInfo);

    return *this;
}

SessionResponseFactory&
SessionResponseFactory::fromInvalidCookieResp(const bb::MultiSessionResp& sInfo) {
    _result.status = getStatus(sInfo);

    return *this;
}

SessionResponseFactory&
SessionResponseFactory::fromCookie(bb::MultiSessionResp& resp, const std::string& currentUid) {
    std::unique_ptr<bb::SessionResp> current;
    bool valid = false;
    if (SessionResponse::succeeded(getStatus(resp))) {
        for (int i = 0; i < resp.count(); i++) {
            auto user = resp.user(i);
            std::string id = resp.id(i);
            if (id == resp.defaultUid() || id == currentUid) {
                current.reset(user.release());
                _defaultPassId = id;
                valid = true;
            }

            if (id == currentUid) {
                break;
            }
        }
    }

    if (valid) {
        return fromValidCookieResp(resp, *current);
    } else {
        return fromInvalidCookieResp(resp);
    }
}

SessionResponseFactory& SessionResponseFactory::fromOAuth(bb::SessionResp& resp) {
    bb::DBFields dbFields(&resp);

    _defaultPassId = getUid(resp);

    _result.mailLoginAllowed = mailLoginAllowed(dbFields.get(DBFieldsNames::mailLoginRule()));
    _result.serviceUserId = dbFields.get(DBFieldsNames::mailServiceUserId());
    _result.status = getStatus(resp);
    _result.hasPassword = hasPassword(resp);
    _result.isMailish = isMailish(resp);
    _result.mailStatus = getSubscriptionMailStatus(resp);
    _result.uid = getUid(resp);
    _result.age = age(resp);
    _result.oAuthInfo = getOAuthInfo(resp);
    _result.isOAuth = true;
    _result.isSSO = isSSO(resp);
    _userTicket = getUserTicket(resp);

    return *this;
}

}

}
