#pragma once

#include <memory>
#include <string>
#include <set>
#include <stdexcept>
#include <future>
#include <mail/akita/service/include/account/featured_account.h>
#include <mail/akita/service/include/config.h>

namespace akita
{

struct CommonCheckRequest {
    std::set<akita::ServiceId> sidsToCheck;
    std::set<akita::ServiceId> attributesToCheck;
    bool deferredEmailListRequest = true;
    std::string authDomain;
    std::string realIp;
    std::string realPort;
};

struct OAuthRequest {
    CommonCheckRequest common;
    std::string token;
};

struct SessionRequestSSL {
    CommonCheckRequest common;
    std::string authHost;
    std::string sessionID;
    std::string currentUid;
    std::string sessGuard;
};

/**
 * Interface which provides operation is needed from the Black box service.
 */
class BlackBox
{
public:

    virtual ~BlackBox() {}

    typedef std::unique_ptr<akita::FeaturedAccount> FeaturedAccountPtr;

    /**
     * Information about user's OAuth token
     */
    struct OAuthInfo {
        std::vector<std::string> scopes;
    };

    /**
     * checkSession response structure which describes
     * additional information returned from BlackBox and
     * is needed during the authorization check procedure.
     */
    struct SessionResponse
    {
        /**
         * This is redefinition of the libblackbox2 constants actually,
         * but this is the only way to provide independence from the original
         * library.
         */
        enum Status
        {
            Status_valid,
            Status_needReset,
            Status_expired,
            Status_noAuth,
            Status_disabled,
            Status_invalid,
            Status_wrongGuard = 8,
        };
        /**
         * Flag describes if mail login is allowed by service subscription
         * policy.
         */
        bool mailLoginAllowed = false;
        /// Account status
        Status status = Status_invalid;
        /// Service user id (sid) forwarded from database request result
        std::string serviceUserId;
        bool hasPassword = false;

        std::string uid;

        time_t age;//password_verification_age gives seconds since last password input

        OAuthInfo oAuthInfo;
        bool isOAuth = false;
        bool isSSO = false;
        bool isMailish = false;
        SubscriptionMailStatus mailStatus = SubscriptionMailStatus::undefined;

        /// Indicates if response succeeded and account information is valid
        static bool succeeded(Status status)
        {
            return status == Status_valid || status == Status_needReset;
        }

    }; // class SessionResponse

    using OnCheck = std::function<void(mail_errors::error_code, SessionResponse, FeaturedAccountPtr)>;

    virtual void checkSessionSSLAsync(SessionRequestSSL request, OnCheck h) const = 0;

    virtual void checkOAuthAsync(OAuthRequest request, OnCheck h) const = 0;

    virtual bool isAllowedOAuthScope(const std::string& scope) const = 0;

}; // class BlackBox

typedef std::shared_ptr<akita::BlackBox> BlackBoxPtr;

BlackBoxPtr makeBlackBox(http_getter::ClientPtr client, const http_getter::Endpoint& e,
                         Logger logger, const AllowedOAuthScopes& scopes);

struct LoginReqData {
    HttpArguments query;
    std::string body;
};

} // namespace akita

