#pragma once


#include <mail/akita/service/include/blackbox/internal/addresses_parser.h>
#include <mail/akita/service/include/blackbox/libblackbox2.h>
#include <mail/akita/service/include/blackbox/blackbox.h>
#include <mail/akita/service/include/account/featured_account.h>
#include <butil/datetime/date_utils.h>
#include <butil/network/idn.h>
#include <butil/network/utils.h>
#include <butil/digest.h>


namespace akita::blackbox {

struct CreateFromXmlResponse {
    using ResponsePtr = std::unique_ptr<bb::Response>;

    CreateFromXmlResponse(ResponsePtr resp, CommonCheckRequest&& request,
                          FeaturedAccount::AccountArray&& childAccounts,
                          std::string&& tvm2UserTicket, std::string&& bbConnectionId,
                          std::string&& bbLoginId)
        : response_(std::move(resp))
        , sidsToCheck_(std::move(request.sidsToCheck))
        , attributesToCheck_(std::move(request.attributesToCheck))
        , account_(std::make_unique<FeaturedAccount>())
    {
        sidsToCheck_.insert(DBFieldsNames::mailServiceId());
        account_->childAccounts = std::move(childAccounts);
        account_->userTicket = std::move(tvm2UserTicket);
        account_->bbConnectionId = std::move(bbConnectionId);
        account_->bbLoginId = std::move(bbLoginId);

        parseDisplayName();
        parseAttributes();
        parseAddresses();
        parseDbFields(request.authDomain);
        parsePortalLogin();
        fillAliases();
    }

    std::unique_ptr<FeaturedAccount> account() {
        return std::move(account_);
    }

private:
    bb::Response* response() const {
        return response_.get();
    }

    void parseDisplayName() const {
        bb::DisplayNameInfo displayNameInfo(response());
        std::optional<SocialInfo> socialInfo;
        if (displayNameInfo.social()) {
            socialInfo = SocialInfo{
                .profile=displayNameInfo.socProfile(),
                .provider=displayNameInfo.socProvider(),
                .target=displayNameInfo.socTarget(),
            };
        }
        account_->displayName = DisplayName {
            .name=displayNameInfo.name(),
            .socialInfo=socialInfo,
            .avatar=displayNameInfo.defaultAvatar()
        };
    }

    void splitDomain(const std::string& addr, std::string& login, std::string& domain) const {
        split_domain(idna::decode_addrspec(addr), login, domain);
    }

    void parseDbFields(const std::string& authDomain) const {
        bb::DBFields dbFields(response());
        fillLoginAndDomainAndKind(dbFields, authDomain);
        fillUserData(dbFields);
        fillServices(dbFields);
    }

    void fillLoginAndDomainAndKind(const bb::DBFields& fields, const std::string& authDomain) const {
        static const std::string corporateDomain = "yandex-team.ru";
        static const std::string corporateAuth = ".yandex-team.ru";

        splitDomain(fields.get(DBFieldsNames::mailLogin()), account_->login, account_->domain);

        if( !account_->domain.empty() ) {
            account_->kind = akita::AccountKind_pdd;
        } else if (authDomain == corporateAuth) {
            account_->kind = akita::AccountKind_corporate;
            account_->domain = corporateDomain;
        } else {
            account_->kind = akita::AccountKind_yandex;

            const bb::AliasList aliasList(response());
            for (const auto& a : aliasList.getAliases()) {
                if (a.type() == bb::AliasList::Item::Mailish) {
                    splitDomain(a.alias(), account_->login, account_->domain);
                }
            }
        }
    }

    void fillAliases() const {
        const bb::AliasList aliasList(response());
        for (const auto& a : aliasList.getAliases()) {
            if (a.type() != bb::AliasList::Item::Mailish) {
                account_->aliases[std::to_string(a.type())] = a.alias();
            }
        }
    }

    void parseAttributes() const {
        bb::Attributes attributes(response());

        for (const auto& attr: attributesToCheck_) {
            if (attributes.has(attr)) {
                account_->requestAttributes[attr] = attributes.get(attr);
            }
        }

        if (attributes.has(Attributes::haveOrganizationNameAttr()) &&
            attributes.get(Attributes::haveOrganizationNameAttr()) == "1") {
            account_->attributes.haveOrganizationName = true;
        }
        if (attributes.has(Attributes::securityLevelAttr())) {
            account_->attributes.securityLevel = attributes.get(Attributes::securityLevelAttr());
        }
        if (attributes.has(Attributes::haveYaplusAttr()) &&
            attributes.get(Attributes::haveYaplusAttr()) == "1") {
            account_->attributes.haveYaplus = true;
        }
        if (attributes.has(Attributes::genderAttr())) {
            account_->gender = attributes.get(Attributes::genderAttr());
        }
    }

    void fillUserData(const bb::DBFields& fields) const {
        account_->serviceUserId = fields.get(DBFieldsNames::mailServiceUserId());
        account_->account = fields.get(DBFieldsNames::mailLogin());

        bb::PDDUserInfo info( response() );
        account_->domainId = info.domId_;

        account_->mailDataBase = fields.get(DBFieldsNames::mailDataBase());
        account_->firstName = fields.get(DBFieldsNames::firstName());
        account_->lastName = fields.get(DBFieldsNames::lastName());
        account_->registrationDate = convertTimeDate(fields.get(DBFieldsNames::registrationDate()));
        account_->country = fields.get(DBFieldsNames::country());
        account_->timeZone = TzInfo(fields.get(DBFieldsNames::timeZone()));
        account_->birthdayDate = fields.get(DBFieldsNames::birthdayDate());
        account_->composeCheck = md5_hex(account_->serviceUserId);
        account_->language = fields.get(DBFieldsNames::userLanguage());
        bb::Uid uid( response() );
        account_->userId = uid.uid();
        bb::KarmaInfo karmaInfo( response() );
        account_->karma.value = karmaInfo.karma();
        account_->karma.status = karmaInfo.karmaStatus();
    }

    void fillServices(const bb::DBFields& fields) const {
        std::copy_if(sidsToCheck_.begin(), sidsToCheck_.end(),
                     std::inserter(account_->services, account_->services.end()),
                     [&](const auto& sid) {
            return !fields.get(DBFieldsNames::serviceUserId(sid)).empty();
        });
    }

    void parseAddresses() const {
        const bb::EmailList emailList( response() );
        account_->defaultAddress = akita::blackbox::parseAddresses(
            emailList.getEmailItems(),
            std::inserter(account_->internal, account_->internal.end()),
            std::inserter(account_->validated, account_->validated.end())
        );
        const bb::AliasList aliasList(response());
        const auto someAlias = akita::blackbox::parseAliases(
            aliasList.getAliases(),
            std::inserter(account_->internal, account_->internal.end())
        );
        if (account_->defaultAddress.empty()) {
            account_->defaultAddress = someAlias;
        }
    }

    void parsePortalLogin() const {
        bb::LoginInfo loginInfo(response());
        account_->portalLogin = loginInfo.login();
    }

    std::time_t convertTimeDate(const std::string& src) const {
        struct tm tms(DateUtils::tmDefault());
        strptime(src.c_str(), "%Y-%m-%d %H:%M:%S", &tms);
        return mktime(&tms);
    }

    ResponsePtr response_;
    std::set<std::string> sidsToCheck_;
    std::set<std::string> attributesToCheck_;
    BlackBox::FeaturedAccountPtr account_;
};

}
