#pragma once

#include <map>
#include <string>
#include <vector>
#include <boost/optional.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <mail/akita/service/include/common/timezone.h>

namespace akita::blackbox::sessionid {

enum class StatusId {
    VALID = 0,
    NEED_RESET = 1,
    EXPIRED = 2,
    NOAUTH = 3,
    DISABLED = 4,
    INVALID = 5
};

struct Status {
    StatusId id = StatusId::INVALID;

    bool isOk() const {
        return id == StatusId::VALID || id == StatusId::NEED_RESET;
    }
};

using Attributes = std::map<std::string, std::string>;

struct User {
    boost::optional<Attributes> attributes;
    Status status;
    std::string uid;
};

struct Response {
    boost::optional<User> defaultUser() const {
        const auto defaultUser = users | boost::adaptors::filtered([this](const User& u) {
            return u.uid == default_uid;
        });

        return defaultUser.empty() ? boost::none : boost::optional<User>(defaultUser.front());
    }

    Status status;
    std::string connection_id;
    std::string default_uid;
    std::string login_id;
    std::vector<User> users;
};

inline std::string timeZoneAttribute() {
    return "33";
}

struct CheckCookies {
    Response response;
    TzInfo info;

    std::string timeZoneFromResponse() const {
        const auto defaultUser = response.defaultUser();

        if (!defaultUser || !defaultUser->attributes) {
            return "";
        }

        const auto attribute = defaultUser->attributes->find(timeZoneAttribute());

        return attribute != defaultUser->attributes->end() ? attribute->second : "";
    }

    CheckCookies(Response r)
        : response(std::move(r))
        , info(timeZoneFromResponse())
    { }

    std::string bbConnectionId() const {
        return response.connection_id;
    }

    std::string bbLoginId() const {
        return response.login_id;
    }

    std::string uid() const {
        return response.default_uid;
    }

    std::vector<std::string> childUids() const {
        using namespace boost::adaptors;

        const auto childUids = response.users | transformed([] (const User& u) {
            return u.uid;
        }) | filtered([this] (const std::string& uid) {
            return uid != response.default_uid;
        });

        return {std::begin(childUids), std::end(childUids)};
    }

    std::string timeZone() const {
        return info.timezone();
    }

    int offset() const {
        return info.offset();
    }
};

} // namespace akita::blackbox::sessionid

