#pragma once

#include <variant>

#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>

#include <mail/unistat/cpp/include/run.h>
#include <mail/unistat/cpp/include/meters/common.h>
#include <mail/unistat/cpp/include/common_logs.h>


namespace akita::metrics {

struct Metric {
    std::map<std::string, std::size_t> accum;

    Metric() = default;

    static std::string fullName(const std::string& version, const std::string& name) {
        return "ctype="+unistat::normalizeName(version)+";"+name;
    }

    void inc(const std::string& version, const std::string& name) {
        const std::string full = fullName(version, name);
        if (accum.find(full) == accum.end()) {
            accum[full] = 1;
        } else {
            accum[full]++;
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        using namespace std::string_literals;
        namespace yr = yamail::data::reflection;

        const auto where = record.find("where_name");
        const auto code = record.find("code");
        const auto reason = record.find("reason");
        const auto yandexuid = record.find("yandexuid");
        const auto source = record.find("source");

        if (yandexuid == record.end() && (where == record.end() || code == record.end() || reason == record.end())) {
            throw std::runtime_error("Line must contain 'where_name' and 'code' and 'reason'");
        }

        const std::string version = where->second+"@"+reason->second+"@"+(source != record.end() ? source->second : "");

        inc(version, "error_code_"s + code->second);
    }

    std::vector<unistat::NamedValue<std::size_t>> get() const {
        std::vector<unistat::NamedValue<std::size_t>> ret;
        ret.reserve(accum.size());

        boost::copy(accum
                    | boost::adaptors::transformed([](const auto& p) {
                        return unistat::NamedValue<std::size_t>{unistat::withSigoptSuffix(p.first, "summ"), p.second};
                    })
                    , std::back_inserter(ret));

        return ret;
    }
};

}

namespace unistat {

using Metric = akita::metrics::Metric;
using LogMeters = std::variant<std::shared_ptr<akita::metrics::Metric>>;
using AkitaLog = Log<TextFileReader, File, TskvParser, LogMeters>;
using AkitaLogPtr = std::shared_ptr<AkitaLog>;
using Logs = std::variant<AkitaLogPtr, HttpClientLogPtr>;

}
