from argparse import ArgumentParser
from mail.unistat.cpp.cython.meters import (
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
)
from collections import namedtuple
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs
import mail.akita.unistat.cpp.run as unistat


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/akita/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='backward compatibility')
    parser.add_argument('akita_config')
    return parser.parse_args()


def remove_scheme(s):
    return s.replace('https://', '').replace('http://', '')


def remove_scheme_and_port(s):
    s = remove_scheme(s)
    i = s.rfind(':')
    if i != -1:
        return s[:i]
    return s


def make_akita_config(data):
    akita = list(filter(lambda m: m['_name'] == 'akita', data['config']['modules']['module']))[0]

    return AkitaConfig(
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        akita_log=os.path.join(os.curdir, data['config']['log']['akita']['sinks'][0]['path']),
        primary=remove_scheme_and_port(akita['configuration']['blackbox']['url']),
        fallback=remove_scheme_and_port(akita['configuration']['blackbox']['fallback']),
    )


def make_akita_http_client_log_meters(cfg):
    ret = [
        HttpClientHttpRequestCountByStatus(cfg.primary, "count_by_status_primary"),
        HttpClientHttpRequestTotalTimeHist((0, 20, 50, 100, 300, 500), cfg.primary, cfg.primary+"_time_hist")
    ]

    if cfg.primary != cfg.fallback:
        ret += [
            HttpClientHttpRequestCountByStatus(cfg.fallback, "count_by_status_fallback"),
            HttpClientHttpRequestTotalTimeHist((0, 20, 50, 100, 300, 500), cfg.fallback, cfg.fallback+"_time_hist"),
        ]

    return ret


def make_metrics_log_meters():
    return [
        unistat.AkitaMetric()
    ]


AkitaConfig = namedtuple('AkitaConfig', (
    'httpclient_log',
    'akita_log',
    'primary',
    'fallback',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.akita_config) as f:
        akita_config = make_akita_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    akita_log = unistat.AkitaMetricsLog(
        [],
        make_metrics_log_meters(),
        fast_forward,
        akita_config.akita_log
    )
    http_client_log = logs.HttpClientLog(
        [],
        make_akita_http_client_log_meters(akita_config),
        fast_forward,
        akita_config.httpclient_log
    )

    unistat.run(args.host, args.port, [akita_log, http_client_log], args.log, logLevel='info')

if __name__ == '__main__':
    main()
