#include <mail/alabay/ymod_logbroker/include/consumer/event_buffer.h>
#include <mail/alabay/ymod_logbroker/include/consumer/event_buffer_proxy.h>
#include <mail/alabay/service/include/log.h>
#include <mail/alabay/service/include/repository.h>
#include <mail/webmail/http_api_helpers/include/find_dependency.h>
#include <yamail/data/serialization/yajl.h>
#include <yplatform/module_registration.h>
#include <yplatform/ptree.h>

namespace alabay {

struct UserJournalEventBuffer: public ymod_logbroker::EventBuffer, public yplatform::module {
    UserJournalEventBuffer(RepositoryPtr repository, OrgRepositoryPtr orgRepository)
        : EventBuffer()
        , repository(std::move(repository))
        , orgRepo(std::move(orgRepository))
    { }

    virtual ~UserJournalEventBuffer() = default;

    void asyncAddEvents(std::string_view, ymod_logbroker::helpers::ParsedEvent parsedLogLine, yplatform::task_context_ptr, ymod_logbroker::OnProbablyFlush hook) override {
        if (parsedLogLine.contains("uid") && parsedLogLine.contains("operation") && parsedLogLine.contains("date")) {
            auto getValueOrNull = [&] (std::string_view key) -> std::optional<std::string> {
                auto it = parsedLogLine.find(key);
                if (parsedLogLine.end() != it) {
                    return std::string(it->second);
                } else {
                    return std::nullopt;
                }
            };

            eventsBuffer.emplace_back(Event {
                .uid=Uid(std::stoll(std::string(parsedLogLine["uid"]))),
                .orgId=OrgId(107908),
                .type=std::string(parsedLogLine["operation"]),
                .date=std::stoull(std::string(parsedLogLine["date"])),
                .requestId=getValueOrNull("requestId"),
            });
        }
        hook(mail_errors::error_code(), false);
    }

    void asyncFlush(yplatform::task_context_ptr, ymod_logbroker::OnFlush hook) override {
        Events events;
        eventsBuffer.swap(events);
        repository->addEvents(CommonParams(), std::move(events), hook);
    }

    std::size_t size() const override {
        return eventsBuffer.size();
    }

    Events eventsBuffer;
    RepositoryPtr repository;
    OrgRepositoryPtr orgRepo;
};

struct UserJournalEventBufferFactory: public ymod_logbroker::EventBufferFactory, public yplatform::module {
    virtual ~UserJournalEventBufferFactory() = default;

    void init(const yplatform::ptree& cfg) {
        repository_ = http_api::findDependency<Repository>(cfg, "dependencies.repository");
        orgRepo_ = http_api::findDependency<OrgRepository>(cfg, "dependencies.org_repository");
        chunkSize_ = cfg.get<std::size_t>("logbroker.events_chunk_size");
        flushTimeout_ = yplatform::time_traits::duration_cast<std::chrono::milliseconds>(
            cfg.get<yplatform::time_traits::duration>("logbroker.flush_timeout")
        );

        consumerName_ = cfg.get<TString>("logbroker.consumer_name");
        yplatform::read_ptree(topics_, cfg.get_child("logbroker"), "topics");

        LOGDOG_(getModuleLogger(), notice, log::message="alabay::UserJournalEventBufferFactory loaded");
    }

    ymod_logbroker::EventBufferPtr create() const override {
        ymod_logbroker::EventBufferPtr eventBuffer = std::make_shared<UserJournalEventBuffer>(repository_, orgRepo_);
        eventBuffer = std::make_shared<ymod_logbroker::WithSize>(std::move(eventBuffer), chunkSize_);
        eventBuffer = std::make_shared<ymod_logbroker::WithTimer>(std::move(eventBuffer), flushTimeout_);
        return eventBuffer;
    }

    const std::vector<TString>& topics() const override {
        return topics_;
    }

    const TString& consumerName() const override {
        return consumerName_;
    }

    std::vector<TString> topics_;
    TString consumerName_;
    RepositoryPtr repository_;
    OrgRepositoryPtr orgRepo_;
    std::chrono::milliseconds flushTimeout_;
    std::size_t chunkSize_;
};

}

DEFINE_SERVICE_OBJECT(alabay::UserJournalEventBufferFactory)
