import os

import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.ctl.lib.main import wrap_main
from mail.pg.auditlogdb.devpack.components.auditlogdb import AuditlogDb
from mail.devpack.lib.components.base import DogAppComponent, FakeRootComponent
from mail.webmail_config.lib.make_config import make_config


class Base(DogAppComponent):
    DEPS = [AuditlogDb]

    def __init__(self, env, components):
        super(Base, self).__init__(env, components, custom_path=f'alabay/{self.NAME}')

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('ymod_db/pg_auditlogdb.conf'), os.path.join(self.config_path, 'pg_auditlogdb.conf'))
        helpers.write2file(resource.find('ymod_db/ch_auditlogdb.yml'), os.path.join(self.config_path, 'ch_auditlogdb.yml'))
        helpers.write2file(resource.find('alabay/tvm_secret'), os.path.join(self.secrets_path, 'tvm_secret'))
        helpers.write2file(resource.find('alabay/ch_password'), os.path.join(self.secrets_path, 'ch_password'))

        helpers.mkdir_recursive(os.path.join(self.get_root(), 'app', 'cache', 'tvm'))
        helpers.mkdir_recursive(os.path.join(self.get_root(), 'app', 'its_client', 'controls'))

        self._generate_config('development')
        replace = {
            'auditlogdb_connection_string': f'host=localhost port={self.components[AuditlogDb].port()} user=auditlog dbname=auditlogdb',
        }
        devpack = self.yhttp.format_config(resource.find(f'alabay_{self.NAME}/local.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.config_path, 'config-devpack.yml'))

        self.yhttp.init_pgpass(self.get_root())

    def _generate_config(self, env):
        base = resource.find(f'alabay_{self.NAME}/config.yml')
        service_yaml = yaml.safe_load(resource.find('alabay/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        auditlogdb_yaml = yaml.safe_load(resource.find('ymod_db/service.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, auditlogdb_yaml, silent=True)
        config = self.yhttp.format_config(cfg)
        helpers.write2file(config, os.path.join(self.config_path, 'config.yml'))

    def start(self):
        super(Base, self).start(put_pgpass_in_env=True)


class HttpApi(Base):
    NAME = 'alabay_http_api'

    def __init__(self, env, components):
        super(HttpApi, self).__init__(env, components)


class Consumer(Base):
    NAME = 'alabay_consumer'
    DEPS = [AuditlogDb]

    def __init__(self, env, components):
        super(Consumer, self).__init__(env, components)


class Service(FakeRootComponent):
    NAME = 'service'
    DEPS = [HttpApi, Consumer]


def main():
    wrap_main(deps_root=Service)
