#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <tvm_guard/module.h>

#include <mail/alabay/service/include/handlers/organization.h>
#include <mail/alabay/service/include/handlers/event.h>
#include <mail/alabay/service/include/types_reflection.h>
#include <mail/alabay/service/include/log.h>

#include <mail/webmail/corgi/include/types_error.h>
#include <mail/webmail/corgi/include/parse_urlencoded.h>
#include <mail/webmail/corgi/include/resolve/directory.h>
#include <mail/webmail/corgi/include/http_api_handlers.h>

#include <mail/webmail/ymod_healthcheck/include/healthcheck.h>

#include <mail/webmail/http_api_helpers/include/context.h>
#include <mail/webmail/http_api_helpers/include/handler_helpers.h>

#include <mail/http_getter/client/include/endpoint_reflection.h>
#include <mail/http_getter/client/include/module.h>

#include <yamail/data/deserialization/ptree_reader.h>
#include <yamail/data/deserialization/json_reader.h>
#include <boost/range/algorithm/transform.hpp>

#include <pa/async.h>


BOOST_FUSION_ADAPT_STRUCT(alabay::OrganizationShowResult, state)
BOOST_FUSION_ADAPT_STRUCT(alabay::EventListResult, data, next)
YREFLECTION_ADAPT_ADT(alabay::EventListParams,
    YREFLECTION_MEMBER_RENAMED(std::time_t, date_from, dateFrom)
    YREFLECTION_MEMBER(corgi::PageParams, page)
)

namespace alabay {

class Response: public http_api::Response {
protected:
    void logSuccessResponse(const std::string& body) const override {
        http_api::Context ctx{stream->request()};

        const auto contextLogger = getContextLogger(
            ctx.optionalArg("adminUid"), ctx.optionalArg("orgId"), ctx.optionalHeader("x-request-id")
        );

        LOGDOG_(contextLogger, notice,
                log::where_name=stream->request()->url.make_full_path(),
                log::message=body);
    }

    void logErrorResponse(const std::string& body) const override {
        http_api::Context ctx{stream->request()};

        const auto contextLogger = getContextLogger(
            ctx.optionalArg("adminUid"), ctx.optionalArg("orgId"), ctx.optionalHeader("x-request-id")
        );

        LOGDOG_(contextLogger, error,
                log::where_name=stream->request()->url.make_full_path(),
                log::message=body);
    }

    bool errorCodeToResponseCode(const mail_errors::error_code& ec, ymod_webserver::codes::code& code) const override {
        if (corgi::errorCodeToResponseCode(ec, code)) {
            return true;
        } else {
            return false;
        }
    }

public:

    Response(ymod_webserver::response_ptr s)
        : http_api::Response(std::move(s))
    { }

    using http_api::Response::with;
    void with(EventListResult&& resp) const {
        defaultJsonResponse(std::move(resp));
    }

    void with(OrganizationShowResult&& resp) const {
        defaultJsonResponse(std::move(resp));
    }
};

template<class F>
auto handler(HttpApiConfigPtr config, F f) {
    return [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
        http_api::Context ctx {s->request()};
        CommonParams common = parseCommonParams(ctx);

        auto uid = std::to_string(common.adminUid.t);
        auto log = http_getter::withLog(config->getter->httpLogger(uid, common.requestId));
        auto httpClient = config->getter->create(*s->request(), log);

        if (!corgi::isAdmin(common, *httpClient, config->resolverConfig, yield).value_or_throw()) {
            throw mail_errors::system_error(make_error(corgi::AccessError::accessDenied, "is not admin"));
        }

        return f(std::move(common), std::move(ctx), std::move(yield));
    };
}

EventListParams parseEventListParams(http_api::Context ctx) {
    return corgi::parse<EventListParams>(ctx);
}

Events parseEvents(http_api::Context ctx) {
    return yamail::data::deserialization::JsonReader<Events>(ctx.body()).result();
}

struct HttpApi: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        http_api::LogArgsAsTskvConfig logArgsCfg {
            .tskvFormat="mail-alabay-args-log",
            .argsLogger=getTskvArgsLogger(cfg.get<std::string>("logger.args"))
        };
        boost::transform(
            cfg.equal_range("params_to_hide"), std::inserter(logArgsCfg.paramsToHide, logArgsCfg.paramsToHide.end()),
            [] (auto&& val) { return val.second.template get<std::string>("name"); }
        );

        http_api::BindInfo<Response> info {
            .guarded=findDependency<tvm_guard::Module>(cfg, "dependencies.guard"),
            .server=findDependency<ymod_webserver::server>(cfg, "dependencies.server"),
            .attributes=boost::coroutines::attributes(cfg.get<unsigned>("coroutine_stack_size")),
            .reactor=findDependency<yplatform::reactor>(cfg, "dependencies.reactor"),
        };

        auto config = std::make_shared<HttpApiConfig> (HttpApiConfig {
            .getter=findDependency<http_getter::TypedClientModule>(cfg, "dependencies.http_getter"),
            .resolverConfig=corgi::fromPtree(cfg),
            .repo=findDependency<Repository>(cfg, "dependencies.repository"),
            .orgRepo=findDependency<OrgRepository>(cfg, "dependencies.org_repository"),
        });

        pa::async_profiler::init(500000, 16, cfg.get<std::string>("profiler_log"));

        organizations(info, logArgsCfg, config);
        user(info, logArgsCfg, config);
        events(info, logArgsCfg, config);

        LOGDOG_(getModuleLogger(), notice, log::message="alabay::HttpApi loaded");
    }

    void organizations(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const HttpApiConfigPtr& config) const {

        bindPOST<OrganizationEnableResult>(info, logArgsCfg, "/v1/organization/enable",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationInit(common, config, yield);
            }));

        bindPOST<OrganizationDisableResult>(info, logArgsCfg, "/v1/organization/disable",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationDisable(common, config, yield);
            }));

        bindGET<OrganizationShowResult>(info, "/v1/organization/show",
            corgi::organizationHandler(config->getter, [=] (OrganizationParams common, http_getter::TypedClientPtr, boost::asio::yield_context yield) {
                return organizationShow(common, config, yield);
            }));
    }

    void user(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const HttpApiConfigPtr& config) const {

        bindPOST<UserSettingResult>(info, logArgsCfg, "/v1/user/enable",
            [=] (ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                http_api::Context ctx {s->request()};
                SettingParams params = parseSettingParams(ctx);

                auto log = http_getter::withLog(config->getter->httpLogger(std::to_string(params.uid.t), params.requestId));
                auto httpClient = config->getter->create(*s->request(), log);

                return userEnable(std::move(params), makeOrgId(ctx.arg("org_id")), std::move(httpClient), config, std::move(yield));
            });

        bindPOST<UserSettingResult>(info, logArgsCfg, "/v1/user/disable",
            corgi::userHandler(config->getter, [=] (SettingParams params, http_getter::TypedClientPtr client, boost::asio::yield_context yield) {
                return userDisable(std::move(params), std::move(client), config, std::move(yield));
            }));
    }

    void events(const http_api::BindInfo<Response>& info, const http_api::LogArgsAsTskvConfig& logArgsCfg, const HttpApiConfigPtr& config) const {

        bindGET<EventListResult>(info, "/v1/event/list",
            handler(config, [=] (CommonParams common, http_api::Context ctx, boost::asio::yield_context yield) {
                return eventList(common, parseEventListParams(ctx), config, yield);
            }));

        bindPOST<EmptyResult>(info, logArgsCfg, "/debug/event/add",
            handler(config, [=] (CommonParams common, http_api::Context ctx, boost::asio::yield_context yield) {
                return eventAdd(common, parseEvents(ctx), config, yield);
            }));
    }

    yplatform::ptree get_stats() const override {
        return ymod_healthcheck::Healthcheck::ready();
    }
};

}

DEFINE_SERVICE_OBJECT(alabay::HttpApi)
