#pragma once

#include <io_result/io_result.h>
#include <io_result/hooks.h>
#include <mail/alabay/service/include/types.h>


namespace alabay {

using OnOrganization = io_result::Hook<io_result::Optional<Organization>>;
using OnEvents = io_result::Hook<io_result::Sequence<Event>>;
using OnExecute = io_result::Hook<>;

struct OrgRepository {
    virtual ~OrgRepository() = default;

    template <typename Handler>
    auto enableOrganization(const OrganizationParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncEnableOrganization(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto getOrganization(const OrganizationParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnOrganization> init(handler);
        asyncGetOrganization(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto disableOrganization(const OrganizationParams& c, std::time_t doomDate, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDisableOrganization(c, doomDate, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto disableUser(const SettingParams& c, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDisableUser(c, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto enableUser(const SettingParams& c, OrgId orgId, const std::string& login, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncEnableUser(c, orgId, login, init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncEnableOrganization(const OrganizationParams&, OnExecute) const = 0;
    virtual void asyncDisableOrganization(const OrganizationParams&, std::time_t, OnExecute) const = 0;
    virtual void asyncGetOrganization(const OrganizationParams&, OnOrganization) const = 0;
    virtual void asyncEnableUser(const SettingParams&, OrgId, const std::string&, OnExecute) const = 0;
    virtual void asyncDisableUser(const SettingParams&, OnExecute) const = 0;
};
using OrgRepositoryPtr = std::shared_ptr<OrgRepository>;

struct Repository {
    virtual ~Repository() = default;

    template <typename Handler>
    auto eventList(const CommonParams& common, const EventListParams& page, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnEvents> init(handler);
        asyncEventList(common, page, init.handler);
        return init.result.get();
    }

    template <typename Handler>
    auto addEvents(const CommonParams& common, Events events, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncAddEvents(common, std::move(events), init.handler);
        return init.result.get();
    }


    template <typename Handler>
    auto addDiskEvents(const CommonParams& common, DiskEvents events, Handler handler) const {
        io_result::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncAddDiskEvents(common, std::move(events), init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncAddEvents(const CommonParams&, Events, OnExecute) const = 0;
    virtual void asyncAddDiskEvents(const CommonParams&, DiskEvents, OnExecute) const = 0;
    virtual void asyncEventList(const CommonParams&, const EventListParams&, OnEvents) const = 0;
};
using RepositoryPtr = std::shared_ptr<Repository>;

}
