#pragma once

#include <string>
#include <boost/asio/spawn.hpp>
#include <yamail/expected.h>
#include <mail/alabay/ymod_logbroker/include/parse_helpers.h>
#include <mail/ymod_queuedb/include/types.h>
#include <mail/webmail/corgi/include/types.h>
#include <spdlog/details/format.h>
#include <mail/alabay/service/include/error.h>


namespace alabay {

using namespace corgi;

using ParsedEvent = ymod_logbroker::helpers::ParsedEvent;

struct Event {
    Uid uid;
    OrgId orgId;
    std::string type;
    std::uint64_t date;
    std::optional<std::string> requestId;
};

using Events = std::vector<Event>;

struct DiskEvent : public Event {
    Uid ownerUid;
    std::string path;
};

using DiskEvents = std::vector<DiskEvent>;

struct UidSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("uid");
        if (line.end() == it) {
            return make_unexpected(ConsumerError::malformedEvent, "required field uid not found");
        }
        std::int64_t result;
        if (!ymod_logbroker::helpers::parseNumber(it->second, result)) {
            return make_unexpected(ConsumerError::malformedEvent, "can't convert uid to number");
        }
        event.uid = Uid(result);
        return event;
    }
};

struct OrgIdSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        event.orgId = 107908;
        return event;
    }
};

struct TypeSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("event_type");
        if (line.end() == it) {
            return make_unexpected(ConsumerError::malformedEvent, "required field event_type not found");
        }
        std::string eventType(it->second);
        if (!supportedEvents.contains(eventType)) {
            return make_unexpected(ConsumerError::unsupportedEventType, "unsupported event type " + eventType);
        }
        event.type = std::move(eventType);
        return event;
    }

    static const inline std::set<std::string> supportedEvents = {"fs-copy", "fs-mkdir", "fs-move", "fs-rm", "fs-set-public", "fs-store",
            "fs-trash-append", "fs-trash-drop", "fs-trash-drop-all", "share-activate-invite", "share-change-rights",
            "share-change-invite-rights", "share-create-group", "share-invite-user"};
};

struct DateSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("tgt_utime");
        if (line.end() == it) {
            return make_unexpected(ConsumerError::malformedEvent, "required field tgt_utime not found");
        }
        if (!ymod_logbroker::helpers::parseNumber(it->second, event.date)) {
            return make_unexpected(ConsumerError::malformedEvent, "can't convert tgt_utime to number");
        }
        return event;
    }
};

struct RequestIdSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("req_id");
        if (line.end() != it) {
            event.requestId = it->second;
        }
        return event;
    }
};

struct OwnerUidSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("owner_uid");
        if (line.end() == it) {
            event.ownerUid = Uid(0);
        } else {
            std::int64_t result;
            if (!ymod_logbroker::helpers::parseNumber(it->second, result)) {
                return make_unexpected(ConsumerError::malformedEvent, "can't convert owner_uid to number");
            }
            event.ownerUid = Uid(result);
        }
        return event;
    }
};

struct PathSetter {
    const ParsedEvent& line;
    yamail::expected<DiskEvent> operator()(DiskEvent event) const {
        auto it = line.find("tgt_rawaddress");
        if (line.end() != it) {
            std::string_view rawPath = it->second;
            const auto pos = rawPath.find_first_of(':');
            if (pos != std::string_view::npos) {
                rawPath.remove_prefix(pos + 1);
                event.path = rawPath;
            }
        }
        return event;
    }
};

struct DiskEventBuilder {
    static yamail::expected<DiskEvent> build(const ParsedEvent& line) {
        return yamail::expected<DiskEvent>(DiskEvent())
                .bind(UidSetter{line})
                .bind(OrgIdSetter{line})
                .bind(TypeSetter{line})
                .bind(DateSetter{line})
                .bind(RequestIdSetter{line})
                .bind(OwnerUidSetter{line})
                .bind(PathSetter{line});

    }
};

struct EventListParams {
    std::time_t dateFrom;
    PageParams page;
};

}
