#include <mail/alabay/service/include/repository.h>
#include <mail/alabay/ymod_db/include/postgres.h>
#include <mail/alabay/service/include/log.h>
#include <mail/webmail/corgi/include/types_error.h>
#include <mail/alabay/service/include/types_reflection.h>
#include <mail/webmail/commondb/include/with_executers.h>
#include <mail/webmail/commondb/include/settings_reflection.h>
#include <mail/webmail/commondb/include/logger.h>
#include <yamail/data/deserialization/ptree.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>
#include <yplatform/module_registration.h>
#include <pgg/database/pool_factory.h>
#include <pgg/factory.h>
#include <fstream>


BOOST_FUSION_DEFINE_STRUCT((alabay), Settings,
    (std::string, reactor)
    (std::string, logger)
)


namespace alabay {
namespace log {
using namespace ::commondb::attr;
}

namespace ydr = yamail::data::reflection;
namespace ydd = yamail::data::deserialization;
using namespace commondb;
using namespace std::string_literals;

namespace {

auto fromReflection(reflection::Organization& data) {
    return alabay::Organization {
        .state = ydr::from_string<OrganizationState>(data.state),
        .doomDate = data.doom_date,
        .created = data.created
    };
}

auto fromReflection(reflection::OptionalOrganization& data) {
    io_result::Optional<alabay::Organization> s;

    if (data.organization) {
        s = fromReflection(*data.organization);
    }
    return s;
}

}

struct Postgres: public WithExecuters, public OrgRepository, public yplatform::module {

    virtual ~Postgres() = default;

    void init(const yplatform::ptree& cfg) {
        common = ydd::fromPtree<Settings>(cfg);
        pg = readPgSettings(cfg.get_child("pg"));

        initExecuters(common.reactor, pg, query::QueriesRegister(), query::ParametersRegister());

        LOGDOG_(getModuleLogger(common.logger), notice,
            log::message="alabay::Postgres module initialized",
            log::settings=yamail::data::serialization::JsonWriter(common).result(),
            log::pg_settings=yamail::data::serialization::JsonWriter(pg).result(),
            log::password_length=passwordSize()
        );
    }

    using WithExecuters::anyExecutor;
    using WithExecuters::readWriteExecutor;

    template<class Params>
    auto master(const Params& c) const {
        auto ctxLogger = getContextLogger(c);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(c.orgId.t), c.requestId, "auditlogdb");
        return readWriteExecutor(c.requestId, logger, profiler);
    }

    template<>
    auto master(const SettingParams& c) const {
        auto ctxLogger = getContextLogger(c);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("-", c.requestId, "auditlogdb");
        return readWriteExecutor(c.requestId, logger, profiler);
    }

    template<class Params>
    auto anything(const Params& c) const {
        auto ctxLogger = getContextLogger(c);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler(std::to_string(c.orgId.t), c.requestId, "auditlogdb");
        return anyExecutor(c.requestId, logger, profiler);
    }

    template<>
    auto anything(const RequestId& requestId) const {
        auto ctxLogger = getContextLogger(std::nullopt, std::nullopt, requestId);
        auto logger = makePggLogger(ctxLogger);
        auto profiler = commondb::pggProfiler("-", requestId, "auditlogdb");
        return anyExecutor(requestId, logger, profiler);
    }

    void asyncEnableOrganization(const OrganizationParams& c, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::EnableOrganization>(
                query::OrgId(c.orgId)
            ),
            std::move(cb)
        );
    }

    void asyncDisableOrganization(const OrganizationParams& c, std::time_t doomDate, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::DisableOrganization>(
                query::OrgId(c.orgId),
                query::DoomDate(doomDate)
            ),
            std::move(cb)
        );
    }

    void asyncGetOrganization(const OrganizationParams& c, OnOrganization cb) const override {
        anything(c)->request(
            pgg::withQuery<query::GetOrganization>(
                query::OrgId(c.orgId)
            ),
            std::move(cb),
            [] (reflection::OptionalOrganization data) { return fromReflection(data); }
        );
    }

    void asyncEnableUser(const SettingParams& c, OrgId orgId, const std::string& login, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::EnableUser>(
                query::OrgId(orgId),
                query::Uid(c.uid),
                query::Login(login)
            ),
            std::move(cb)
        );
    }

    void asyncDisableUser(const SettingParams& c, OnExecute cb) const override {
        master(c)->execute(
            pgg::withQuery<query::DisableUser>(
                query::Uid(c.uid)
            ),
            std::move(cb)
        );
    }

    Settings common;
    PgSettings pg;
};

}

DEFINE_SERVICE_OBJECT(alabay::Postgres)
