#include <mail/alabay/ymod_logbroker/include/consumer/consumer.h>
#include <mail/alabay/ymod_logbroker/include/consumer/metrics.h>
#include <mail/alabay/ymod_logbroker/include/consumer/log.h>
#include <mail/alabay/ymod_logbroker/include/log.h>
#include <mail/tvm_guard/tvm_api/log_tskv.h>
#include <kikimr/public/sdk/cpp/client/ydb_persqueue/persqueue.h>
#include <kikimr/public/sdk/cpp/client/tvm/tvm.h>

#include <mail/webmail/tskv/parser.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/env.h>

namespace ymod_logbroker {

std::shared_ptr<NTvmAuth::TTvmClient> makeTvmClient(ConsumerConfigPtr config) {
    auto log = tvm_guard::makeLogger(config->logger);
    return std::make_shared<NTvmAuth::TTvmClient>(config->tvmClientSettings, std::move(log));
}

struct DataEventProcessor {
    explicit DataEventProcessor(bool commitAfterProcessing, EventBufferPtr buffer,
                                ModuleLogger l, ConsumerLogger c)
        : metrics(l)
        , commitAfterProcessing(commitAfterProcessing)
        , eventBuffer(std::move(buffer))
        , logger(l)
        , consumerLog(c)
    { }

    using TDataReceivedEvent = NYdb::NPersQueue::TReadSessionEvent::TDataReceivedEvent;

    void onEvent(std::optional<NYdb::NPersQueue::TReadSessionEvent::TDataReceivedEvent> eventOpt) {
        if (!eventOpt) {
            if (eventBuffer->probablyFlush({}, io_result::use_sync)) {
                commitMessages();
            }
            return;
        }
        auto event = *std::move(eventOpt);
        metrics.dataReceived(event);
        auto ctx = boost::make_shared<yplatform::task_context>();
        ctx->begin();
        onMessages(event.GetMessages(), ctx);
        if (commitAfterProcessing) {
            eventsForCommit.emplace_back(std::move(event));
        }
    }

    void commitMessages() {
        for (auto& e : eventsForCommit) {
            e.Commit();
        }
        eventsForCommit.clear();
    }

    void onMessages(const TVector<TDataReceivedEvent::TMessage>& msgs, yplatform::task_context_ptr ctx) {
        for (const auto& msg : msgs) {
            try {
                std::string_view line = msg.GetData();
                if (auto parsed = webmail::tskv::parseInplace(line); parsed) {
                    if (eventBuffer->addEvents(line, std::move(parsed->parsed), ctx, io_result::use_sync)) {
                        commitMessages();
                        consumerProcessed(consumerLog, ctx, msgs);
                    }
                } else {
                    metrics.cannotProcessMessage();
                    LOGDOG_(logger, error, log::message="Can't parse: " + msg.GetData());
                }
            } catch (const std::exception& e) {
                consumerFailed(consumerLog, ctx, msgs, e);
                metrics.cannotProcessMessage();
                LOGDOG_(logger, error, log::message="Data: " + msg.GetData(), log::exception=e);
            }
        }
    }

private:
    const ConsumerMetrics metrics;
    bool commitAfterProcessing;
    std::vector<TDataReceivedEvent> eventsForCommit;
    EventBufferPtr eventBuffer;
    ModuleLogger logger;
    ConsumerLogger consumerLog;
};

struct LogbrokerConsumer {
    LogbrokerConsumer(ConsumerConfigPtr config, std::shared_ptr<NYdb::ICredentialsProviderFactory> tvmProvider,
                      ModuleLogger moduleLogger)
        : driverConfig(createDriverConfig(config, std::move(tvmProvider)))
        , driver(driverConfig)
        , persqueueClient(driver)
        , sessionSettings(createSessionSettings(config))
        , readSession(createReadSession())
        , metrics(moduleLogger)
        , waitDuration(config->waitDuration)
        , logger(moduleLogger)
    {}


    std::optional <NYdb::NPersQueue::TReadSessionEvent::TDataReceivedEvent> getDataEvent(const TaskContextHolder& taskContextHolder) {

        try {
            using namespace NYdb::NPersQueue;

            auto future = readSession->WaitEvent();
            future.Wait(waitDuration);

            const bool blockIfNoEventReceivedYet = true;
            TMaybe<TReadSessionEvent::TEvent> eventOpt = readSession->GetEvent(blockIfNoEventReceivedYet);
            if (taskContextHolder.cancelled()) {
                readSession->Close();
                return std::nullopt;
            }
            TReadSessionEvent::TEvent event = *std::move(eventOpt);

            if (std::holds_alternative<TReadSessionEvent::TDataReceivedEvent>(event)) {
                return std::get<TReadSessionEvent::TDataReceivedEvent>(std::move(event));
            } else if (auto* createPartitionStreamEvent = std::get_if<TReadSessionEvent::TCreatePartitionStreamEvent>(&event)) {
                createPartitionStreamEvent->Confirm();
                metrics.createPartitionStream();
            } else if (auto* destroyPartitionStreamEvent = std::get_if<TReadSessionEvent::TDestroyPartitionStreamEvent>(&event)) {
                destroyPartitionStreamEvent->Confirm();
                metrics.destroyPartitionStream();
            } else if (const auto* closeSessionEvent = std::get_if<TSessionClosedEvent>(&event)) {
                metrics.closeSession();
                readSession = createReadSession();
            }
        } catch (const std::exception& e) {
            LOGDOG_(logger, error, log::exception=e);
        } catch (...) {
            LOGDOG_(logger, error, log::message="unknown exception");
        }

        return std::nullopt;
    }

private:
    static NYdb::TDriverConfig createDriverConfig(ConsumerConfigPtr config, std::shared_ptr<NYdb::ICredentialsProviderFactory> tvmProvider) {
        auto driverConfig = NYdb::TDriverConfig()
            .SetNetworkThreadsNum(2)
            .SetEndpoint(config->logbrokerEndpoint)
            .SetDatabase(config->logbrokerDatabase)
            .SetCredentialsProviderFactory(std::move(tvmProvider))
            .SetLog(CreateLogBackend(config->driverLogPath, ELogPriority::TLOG_NOTICE));

        if (config->useSecureConnection) {
            driverConfig.UseSecureConnection();
        }

        return driverConfig;
    }

    static NYdb::NPersQueue::TReadSessionSettings createSessionSettings(ConsumerConfigPtr config) {
        NYdb::NPersQueue::TReadSessionSettings settings;
        settings.ConsumerName(config->eventBufferFactory->consumerName());
        for (const auto& topic : config->eventBufferFactory->topics()) {
            settings.AppendTopics(topic);
        }

        settings.DisableClusterDiscovery(config->disableClusterDiscovery);

        return settings;
    }

    std::shared_ptr<NYdb::NPersQueue::IReadSession> createReadSession() {
        return persqueueClient.CreateReadSession(sessionSettings);
    }

    NYdb::TDriverConfig driverConfig;
    NYdb::TDriver driver;
    NYdb::NPersQueue::TPersQueueClient persqueueClient;
    NYdb::NPersQueue::TReadSessionSettings sessionSettings;
    std::shared_ptr<NYdb::NPersQueue::IReadSession> readSession;
    const ConsumerMetrics metrics;
    TDuration waitDuration;
    ModuleLogger logger;
};

bool waitToContinue(ymod_switchbox::ModuleProviderPtr switchbox) {
    return !switchbox->waitExpectedValue("pause_worker", false, io_result::use_sync);
}

void consume(ConsumerConfigPtr config, TaskContextHolder& taskContextHolder) {
    std::shared_ptr<NTvmAuth::TTvmClient> tvmClient = makeTvmClient(config);
    std::shared_ptr<NYdb::ICredentialsProviderFactory> tvmProvider =
            NYdb::CreateTVMCredentialsProviderFactory(tvmClient, config->tvmService);

    LogbrokerConsumer logbrokerConsumer(config, tvmProvider, config->logger);
    DataEventProcessor processor(
        config->commitAfterProcessing,
        config->eventBufferFactory->create(), config->logger, config->consumerLog
    );

    while (waitToContinue(config->switchbox) && !taskContextHolder.cancelled()) {
        try {
            processor.onEvent(logbrokerConsumer.getDataEvent(taskContextHolder));
        } catch (const std::exception& e) {
            LOGDOG_(config->logger, error, log::exception=e);
            return;
        } catch (...) {
            LOGDOG_(config->logger, error, log::message="unknown exception");
            return;
        }
    }
}

}
