#include <mail/alabay/ymod_logbroker/include/consumer/event_buffer.h>
#include <mail/alabay/service/include/error.h>

namespace ymod_logbroker {

struct EventBufferProxy : public EventBuffer {
    EventBufferProxy(EventBufferPtr ptr) : ptr_(ptr) {}

    std::size_t size() const override {
        return ptr_->size();
    }

protected:
    virtual bool needFlush() const = 0;

    void asyncProbablyFlush(yplatform::task_context_ptr ctx, OnProbablyFlush handler) override {
        if (needFlush()) {
            flush(ctx, [handler](const mail_errors::error_code& ec) {
                handler(ec, !ec);
            });
        } else {
            ptr_->probablyFlush(ctx, handler);
        }
    }
    void asyncAddEvents(std::string_view rawLine, helpers::ParsedEvent parsedLogLine,
                        yplatform::task_context_ptr ctx, OnProbablyFlush handler) override {
        ptr_->addEvents(rawLine, parsedLogLine, ctx, handler);
    }
    void asyncFlush(yplatform::task_context_ptr ctx, OnFlush handler) override {
        ptr_->flush(ctx, handler);
    }

    EventBufferPtr ptr_;
};

struct WithSize : public EventBufferProxy, public std::enable_shared_from_this<WithSize> {
    WithSize(EventBufferPtr ptr, std::size_t bufferSize)
    : EventBufferProxy(ptr)
    , bufferSize_(bufferSize) {}

    virtual ~WithSize() = default;

protected:
    bool needFlush() const override {
        return ptr_->size() >= bufferSize_;
    }
    void asyncAddEvents(std::string_view rawLine, helpers::ParsedEvent parsedLogLine,
                        yplatform::task_context_ptr ctx, OnProbablyFlush handler) override {
        ptr_->addEvents(rawLine, parsedLogLine, ctx,
        [self=shared_from_this(), ctx, handler](const mail_errors::error_code& ec, bool flushed) {
            if (!ec) {
                self->probablyFlush(ctx, handler);
            } else {
                handler(ec, flushed);
            }
        });
    }

private:
    std::size_t bufferSize_;
};

struct WithTimer : public EventBufferProxy, public std::enable_shared_from_this<WithTimer> {

    using clock = std::chrono::steady_clock;
    using duration = std::chrono::milliseconds;
    using time_point = std::chrono::time_point<clock, duration>;

    WithTimer(ymod_logbroker::EventBufferPtr ptr, duration flushInterval)
        : EventBufferProxy(ptr)
        , flushInterval_(flushInterval)
        , nextFlush_(now() + flushInterval)
        { }

    WithTimer(ymod_logbroker::EventBufferPtr ptr, duration flushInterval, time_point nextFlush)
        : EventBufferProxy(ptr)
        , flushInterval_(flushInterval)
        , nextFlush_(nextFlush)
        { }

    virtual ~WithTimer() = default;

protected:
    bool needFlush() const override {
        return now() >= nextFlush_;
    }
    void asyncFlush(yplatform::task_context_ptr ctx, OnFlush handler) override {
        ptr_->flush(ctx, [handler, self=shared_from_this()] (const mail_errors::error_code& ec) {
            handler(ec);
            if(!ec) {
                self->nextFlush_ = self->now() + self->flushInterval_;
            }
        });
    }

private:
    virtual time_point now() const {
        return std::chrono::time_point_cast<duration>(clock::now());
    }
    const duration flushInterval_;
    time_point nextFlush_;
};

}
