#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <mail/alabay/ymod_logbroker/include/consumer/consumer.h>
#include <mail/alabay/ymod_logbroker/include/producer/producer.h>
#include <mail/alabay/ymod_logbroker/include/consumer/log.h>
#include <mail/alabay/ymod_logbroker/include/log.h>
#include <mail/alabay/ymod_logbroker/include/task_context_holder.h>

#include <mail/webmail/ymod_healthcheck/include/healthcheck.h>
#include <mail/webmail/ymod_healthcheck/include/utils.h>

#include <mail/webmail/http_api_helpers/include/find_dependency.h>

#include <mail/tvm_guard/tvm_api/helpers.h>

namespace ymod_logbroker {

struct Consumer: public yplatform::module {

    TaskContextHolder running_;
    std::shared_ptr<yplatform::reactor> reactor_;
    ConsumerConfigPtr config_;
    ymod_healthcheck::Counter aliveThreadsCounter_;

    void init(const yplatform::ptree& cfg) {
        using namespace http_api;

        config_ = std::make_shared<ConsumerConfig> (ConsumerConfig {
            .logbrokerEndpoint=cfg.get<TString>("logbroker.endpoint"),
            .logbrokerDatabase=cfg.get<TString>("logbroker.database"),
            .driverLogPath=cfg.get<TString>("logbroker.driver_log_path"),
            .tvmService=cfg.get<TString>("logbroker.tvm_service"),
            .tvmClientSettings=tvm_guard::parseSettings(cfg.get_child("tvm_client"), tvm_guard::secretFromFile),
            .waitDuration=TDuration::MilliSeconds(
                yplatform::time_traits::duration_cast<std::chrono::milliseconds>(
                    cfg.get<yplatform::time_traits::duration>("logbroker.wait_events_dur")
                ).count()
            ),
            .commitAfterProcessing=cfg.get<bool>("logbroker.commit_after_processing"),
            .useSecureConnection=cfg.get<bool>("logbroker.use_secure_connection"),
            .disableClusterDiscovery=cfg.get<bool>("logbroker.disable_cluster_discovery"),
            .eventBufferFactory=findDependency<EventBufferFactory>(cfg, "dependencies.event_buffer_factory"),
            .logger=getModuleLogger(cfg.get<std::string>("dependencies.logger")),
            .consumerLog=getConsumerLogger(cfg.get<std::string>("dependencies.access_logger")),
            .switchbox=findDependency<ymod_switchbox::ModuleProvider>(cfg, "dependencies.switchbox"),
        });

        reactor_ = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");
        aliveThreadsCounter_ = 0;

        LOGDOG_(config_->logger, notice, log::message="ymod_logbroker::Consumer loaded");
    }

    yplatform::ptree get_stats() const override {
        return aliveThreadsCounter_ == reactor_->size() ? ymod_healthcheck::Healthcheck::ready()
                                                        : ymod_healthcheck::Healthcheck::dead()
        ;
    }

    void start() {
        for (size_t i = 0; i < reactor_->size(); i++) {
            reactor_->io()->post([this, self=shared_from_this()] () {
                ymod_healthcheck::CounterGuard guard(this->aliveThreadsCounter_);
                consume(config_, running_);
            });
        }
    }

    void stop() {
        running_.cancel();
    }
};

struct Producer: public InputOutputBuffer, public yplatform::module {

    TaskContextHolder running_;
    std::shared_ptr<yplatform::reactor> reactor_;
    ProducerConfigPtr config_;
    ymod_healthcheck::Counter aliveThreadsCounter_;

    virtual ~Producer() = default;

    void init(const yplatform::ptree& cfg) {
        using namespace http_api;

        config_ = std::make_shared<ProducerConfig> (ProducerConfig {
            .common = CommonConfig {
                .logbrokerEndpoint=cfg.get<TString>("logbroker.endpoint"),
                .logbrokerDatabase=cfg.get<TString>("logbroker.database"),
                .driverLogPath=cfg.get<TString>("logbroker.driver_log_path"),
                .tvmService=cfg.get<TString>("logbroker.tvm_service"),
                .tvmClientSettings=tvm_guard::parseSettings(cfg.get_child("tvm_client"), tvm_guard::secretFromFile),
                .useSecureConnection=cfg.get<bool>("logbroker.use_secure_connection"),
                .disableClusterDiscovery=cfg.get<bool>("logbroker.disable_cluster_discovery"),
                .logger=getModuleLogger(cfg.get<std::string>("dependencies.logger")),
            },
            .producerLogger=getProducerLogger(cfg.get<std::string>("dependencies.producer_logger")),
            .topicName=cfg.get<TString>("topic_name")
        });

        reactor_ = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");
        aliveThreadsCounter_ = 0;

        LOGDOG_(config_->common.logger, notice, log::message="ymod_logbroker::Producer loaded");
    }

    yplatform::ptree get_stats() const override {
        return aliveThreadsCounter_ == reactor_->size() ? ymod_healthcheck::Healthcheck::ready()
                                                        : ymod_healthcheck::Healthcheck::dead()
        ;
    }

    void start() {
        for (size_t i = 0; i < reactor_->size(); i++) {
            reactor_->io()->post([this, self=shared_from_this()] () {
                ymod_healthcheck::CounterGuard guard(this->aliveThreadsCounter_);
                produce(*config_, *this, running_);
            });
        }
    }

    void stop() {
        running_.cancel();
    }
};

}

DEFINE_SERVICE_OBJECT(ymod_logbroker::Consumer)
DEFINE_SERVICE_OBJECT(ymod_logbroker::Producer)
