#include <mail/alabay/ymod_logbroker/include/producer/producer.h>
#include <mail/alabay/ymod_logbroker/include/producer/log.h>
#include <mail/alabay/ymod_logbroker/include/session_provider.h>

#include <util/generic/overloaded.h>
#include <util/system/env.h>

namespace ymod_logbroker {

using namespace NYdb::NPersQueue;

bool taskIsAlive(const yplatform::task_context& ctx, const TaskContextHolder& holder) {
    return !ctx.is_cancelled() && !holder.cancelled();
}

struct ProducerLoop {
    NYdb::NPersQueue::IWriteSession& writeSession;
    std::optional<TContinuationToken>& continuationToken;
    const TaskContextHolder& taskContextHolder;
    const Task& task;
    const ui64 initSeqNo;
    ui64 messagesAcked;
    ui64 seqNo;
    bool sessionIsActive;
    Task::Messages::const_iterator current;
    ModuleLogger logger;

    ProducerLoop(NYdb::NPersQueue::IWriteSession& session, std::optional<TContinuationToken>& token,
                 const Task& task, const TaskContextHolder& holder, ModuleLogger logger)
        : writeSession(session)
        , continuationToken(token)
        , taskContextHolder(holder)
        , task(task)
        , initSeqNo(writeSession.GetInitSeqNo().GetValueSync())
        , messagesAcked(0)
        , seqNo(initSeqNo + 1)
        , sessionIsActive(true)
        , current(task.messages->begin())
        , logger(std::move(logger))
    { }

    bool notAllEventsAreAcked() const {
        return messagesAcked < task.messages->size();
    }

    bool goAhead() const {
        return notAllEventsAreAcked() && sessionIsActive && taskIsAlive(*task.ctx, taskContextHolder);
    }

    bool blockingReadEventsUntilContinuationTokenIsRead() const {
        return !continuationToken && goAhead();
    }

    bool processEventsWhileContinuationTokenIsSet() const {
        return continuationToken && goAhead();
    }

    void processEvent(TWriteSessionEvent::TEvent& event) {
        std::visit(TOverloaded {
            [this] (const TWriteSessionEvent::TAcksEvent& event) {
                LOGDOG_(logger, debug, log::message=event.DebugString());
                messagesAcked += event.Acks.size();
            },
            [this] (TWriteSessionEvent::TReadyToAcceptEvent& event) {
                LOGDOG_(logger, debug, log::message=event.DebugString());
                continuationToken = std::move(event.ContinuationToken);
            },
            [this] (const TSessionClosedEvent& event) {
                LOGDOG_(logger, debug, log::message=event.DebugString());
                sessionIsActive = false;
            }
        }, event);
    }

    yamail::expected<void> run() {
        while (goAhead()) {
            while (blockingReadEventsUntilContinuationTokenIsRead()) {
                TWriteSessionEvent::TEvent event = *writeSession.GetEvent(true);
                processEvent(event);
            }

            while (processEventsWhileContinuationTokenIsSet()) {
                if (current != task.messages->end()) {
                    writeSession.Write(std::move(*continuationToken), *current++, seqNo++, Now());
                    LOGDOG_(logger, debug, log::message=fmt::format("Message Written: seqNo={}", seqNo));
                    continuationToken = std::nullopt;
                }

                for (TWriteSessionEvent::TEvent& event: writeSession.GetEvents(false)) {
                    processEvent(event);
                }
            }
        }

        LOGDOG_(logger, debug, log::message="Task is done");

        if (!sessionIsActive) {
            return make_unexpected(ProduceError::sessionClosed);
        } else if (!taskIsAlive(*task.ctx, taskContextHolder)) {
            return make_unexpected(ProduceError::cancelled);
        } else if (notAllEventsAreAcked()) {
            return make_unexpected(ProduceError::internalError, "strange state: loop is finished, but not all events are acked");
        } else {
            return {};
        }
    }
};

struct LogbrokerProducer {
    LogbrokerProducer(SessionProvider& provider, std::optional<TContinuationToken>& token, const ProducerConfig& cfg)
        : provider(provider)
        , continuationToken(token)
        , config(cfg)
    { }

    void produce(const Task& task, const TaskContextHolder& taskContextHolder) {
        if (!session) {
            session = provider.createSession(config);
            continuationToken = std::nullopt;
        }

        processResult(task, runLoop(task, taskContextHolder));
    }

    yamail::expected<void> runLoop(const Task& task, const TaskContextHolder& taskContextHolder) const {
        yamail::expected<void> result;
        try {
            result = ProducerLoop(*session, continuationToken, task, taskContextHolder, config.common.logger).run();
        } catch (const mail_errors::system_error& ex) {
            result = yamail::make_unexpected(mail_errors::error_code(ex.code()));
        } catch (const std::exception& ex) {
            result = make_unexpected(ProduceError::internalError, ex.what());
        }

        return result;
    }

    void processResult(const Task& task, const yamail::expected<void>& result) {
        if (result) {
            task.hook();
            producerLogSuccess(config.producerLogger, task);
        } else {
            mail_errors::error_code ec = result.error();

            if (ec.category() != getProduceCategory()) {
                ec = make_error(ProduceError::internalError, ec.full_message());
            }

            switch (static_cast<ProduceError>(ec.value())) {
                case ProduceError::sessionClosed:
                    session.reset();
                    [[fallthrough]];
                case ProduceError::cancelled:
                    [[fallthrough]];
                case ProduceError::internalError:
                    task.hook(ec);
                    producerLogFailed(config.producerLogger, task, ec);
                break;
            }
        }
    }

    SessionProvider& provider;
    std::optional<TContinuationToken>& continuationToken;
    const ProducerConfig& config;
    std::shared_ptr<NYdb::NPersQueue::IWriteSession> session;
};

void produce(const ProducerConfig& config, InputTaskBuffer& taskBuffer, const TaskContextHolder& taskContextHolder) {
    SessionProvider sessionProvider(config.common, makeTvmProvider(config));
    std::optional<TContinuationToken> continuationToken;
    LogbrokerProducer producer(sessionProvider, continuationToken, config);

    while (!taskContextHolder.cancelled()) {
        producer.produce(taskBuffer.popFront(), taskContextHolder);
    }
}

}
