#!/usr/bin/env python

from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf,
    cli
)

from qb2.api.v1 import (
    extractors as qe,
    filters as qf,
    QB2
)

from qb2.api.v1.typing import (
    Optional, Integer,
    String, List,
    Dict, Float, Bool
)


def str_to_int(txt):
    try:
        return int(txt)
    except:
        return None


def extract_uid(keys, values):
    try:
        return dict(zip(eval(keys), eval(values)))["uid"]
    except:
        return None


def get_mid_from_ios_event_value(event_value):
    if event_value["new_value"] == "appeared":
        for sender in event_value["sender"]:
            if sender["class"] == "YOMessageGroupViewController":
                return sender["info"]["YOMessageGroupViewController@mids"]

    return None


def get_key(json_dict, key):
    try:
        return json_dict[key]
    except:
        return None


def fill_mid_with_tid(mid, tid):
    if mid is None:
        return tid
    return mid


def extract_exps(keys, values):
    int_exps = []

    try:
        for str_exp in eval(dict(zip(eval(keys), eval(values)))["experiments"]):
            int_exps.append(int(str_exp))
        return int_exps

    except:
        return int_exps


def extract_mail_provider(keys, values):
    try:
        return dict(zip(eval(keys), eval(values)))["mail_provider"]
    except:
        return None


def prep_open_liza_and_quinn(job):
    return job.table("//home/mail-logs/core/mail-user-events/1d/@dates") \
        .qb2(log="generic-yson-log",
             fields=[qe.integer_log_field("uid").add_hints(type=Optional[Integer]),
                     qe.log_field("operation").hide(),
                     qe.log_field("target").hide(),
                     qe.log_field("stateJson").hide(),
                     qe.log_field("client").add_hints(type=Optional[String]),
                     qe.log_field("testIds").rename("exps").add_hints(type=Optional[List[Integer]]),
                     qe.custom("mid", lambda stateJson: str_to_int(get_key(stateJson, "mid"))).add_hints(
                         type=Optional[Integer]),
                     qe.custom("event_datetime",
                               lambda stateJson: get_key(stateJson, "iso_eventtime")).add_hints(type=Optional[String]),
                     ],
             filters=[qf.equals("operation", "startReading"),
                      qf.equals("target", "message")
                      ]) \
        .filter(qf.defined("uid", "mid", "client", "event_datetime", "exps"),
                qf.compare("uid", ">", 0),
                qf.compare("mid", ">", 0),
                qf.not_(qf.equals("event_datetime", "")))


def prep_open_android(job):
    return job.table("//home/mail-logs/mobile/appmetrika/android/@dates") \
        .qb2(log="generic-yson-log",
             fields=[qe.log_field("EventName").hide(),
                     qe.json_log_field("EventValue").hide(),
                     qe.custom("class", lambda EventValue: get_key(EventValue, "class")).hide(),
                     qe.custom("raw_mid", lambda EventValue: str_to_int(get_key(EventValue, "mid"))).add_hints(type=Optional[Integer]),
                     qe.custom("raw_tid", lambda EventValue: str_to_int(get_key(EventValue, "tid"))).add_hints(type=Optional[Integer]),
                     qe.log_field("ReportEnvironment_Keys").rename("env_keys").hide(),
                     qe.log_field("ReportEnvironment_Values").rename("env_values").hide(),
                     qe.custom("uid",
                               lambda env_keys, env_values: str_to_int(extract_uid(env_keys, env_values))).add_hints(
                         type=Optional[Integer]),
                     qe.custom("provider",
                               lambda env_keys, env_values: extract_mail_provider(env_keys, env_values)).add_hints(
                         type=Optional[String]),
                     qe.custom("exps",
                               lambda env_keys, env_values: extract_exps(env_keys, env_values)).add_hints(
                         type=Optional[List[Integer]]),
                     qe.const("opened", True).add_hints(type=Optional[Bool]),
                     qe.const("client", "Android").add_hints(type=Optional[String]),
                     qe.log_field("EventDateTime").rename("event_datetime").add_hints(type=Optional[String]),
                     qe.integer_log_field("AppBuildNumber").rename("app_build").add_hints(type=Optional[Integer]),
                     qe.log_field("DeviceID").rename("device_id").add_hints(type=Optional[String]),
                     qe.log_field("UUID").rename("uuid").add_hints(type=Optional[String])
                     ],
             filters=[qf.equals("EventName", "dynametric_view_click"),
                      qf.equals("class", "class com.yandex.mail.MessageListItemView")
                      ]) \
        .project(
            uid="uid",
            mid=ne.custom(lambda raw_mid, raw_tid: fill_mid_with_tid(raw_mid, raw_tid), "raw_mid", "raw_tid").add_hints(type=Optional[Integer]),
            client="client",
            event_datetime="event_datetime",
            exps="exps",
            app_build="app_build",
            device_id="device_id",
            uuid="uuid",
            provider="provider"
        ) \
        .filter(qf.defined("uid", "mid", "client", "event_datetime", "exps", "app_build",
                           "device_id", "uuid"),
                qf.compare("uid", ">", 0),
                qf.compare("mid", ">", 0),
                qf.compare("app_build", ">", 0),
                qf.not_(qf.equals("device_id", "")),
                qf.not_(qf.equals("uuid", "")),
                qf.not_(qf.equals("event_datetime", "")))


def prep_open_ios(job):
    return job.table("//home/mail-logs/mobile/appmetrika/ios/@dates") \
        .qb2(log="generic-yson-log",
             fields=[qe.log_field("EventName").hide(),
                     qe.json_log_field("EventValue").hide(),
                     qe.log_field("ReportEnvironment_Keys").rename("env_keys").hide(),
                     qe.log_field("ReportEnvironment_Values").rename("env_values").hide(),
                     qe.custom("uid",
                               lambda env_keys, env_values: str_to_int(extract_uid(env_keys, env_values))).add_hints(
                         type=Optional[Integer]),
                     qe.custom("provider",
                               lambda env_keys, env_values: extract_mail_provider(env_keys, env_values)).add_hints(
                         type=Optional[String]),
                     qe.custom("exps",
                               lambda env_keys, env_values: extract_exps(env_keys, env_values)).add_hints(
                         type=Optional[List[Integer]]),
                     qe.custom("mids", get_mid_from_ios_event_value, "EventValue").hide(),
                     qe.unfold("mid", "mids").add_hints(type=Optional[Integer]),
                     qe.const("client", "iOS").add_hints(type=Optional[String]),
                     qe.log_field("EventDateTime").rename("event_datetime").add_hints(type=Optional[String]),
                     qe.integer_log_field("AppBuildNumber").rename("app_build").add_hints(type=Optional[Integer]),
                     qe.log_field("DeviceID").rename("device_id").add_hints(type=Optional[String]),
                     qe.log_field("UUID").rename("uuid").add_hints(type=Optional[String])
                     ],
             filters=[qf.equals("EventName", "dynametric_uiviewcontroller_did_change_view_state"),
                      qf.defined("uid", "mid", "client", "event_datetime", "exps", "app_build",
                                 "device_id", "uuid"),
                      qf.compare("uid", ">", 0),
                      qf.compare("mid", ">", 0),
                      qf.compare("app_build", ">", 0),
                      qf.not_(qf.equals("device_id", "")),
                      qf.not_(qf.equals("uuid", "")),
                      qf.not_(qf.equals("event_datetime", ""))
                      ])


def prep_opened_emails(job):

    return job.concat(prep_open_android(job),
                      prep_open_ios(job),
                      prep_open_liza_and_quinn(job))


@cli.statinfra_job
def make_job(job, nirvana, statface_client):
    output_table = nirvana.output_tables[0]
    prep_opened_emails(job).put(output_table)
    return job


if __name__ == '__main__':
    cli.run()
