#include <apq/connection_pool.hpp>

#include <boost/asio/io_service.hpp>

#include <iostream>

namespace asio = boost::asio;

struct request_operation
{
    boost::asio::io_context& io;
    const apq::query& query;
    apq::connection_pool& connection_pool;
    apq::time_traits::duration_type request_timeout;

    void run()
    {
        connection_pool.async_request(
            query,
            [this](const auto& res, auto it) { handle_request(res, it); },
            apq::result_format_binary,
            request_timeout);
    }

    void handle_request(const apq::result& res, apq::row_iterator it)
    {
        if (res)
        {
            throw std::runtime_error(res.message());
        }

        std::cout << "Selected:" << std::endl;
        for (; it != apq::row_iterator(); ++it)
        {
            std::int64_t value;
            it->at(0, value);
            std::cout << value << std::endl;
        }
    }
};

int main(int argc, char* argv[])
{
    if (argc < 2)
    {
        std::cout << "Usage: " << argv[0] << " <conninfo>" << std::endl;
        return 1;
    }

    std::cout << "apq request example" << std::endl;

    asio::io_context io(1);
    apq::connection_pool connection_pool(io);
    connection_pool.set_conninfo(argv[1]);
    connection_pool.set_limit(1);
    connection_pool.set_connect_timeout(std::chrono::seconds(3));
    connection_pool.set_queue_timeout(std::chrono::seconds(3));
    auto query = apq::query("SELECT $1::bigint + $2::bigint");
    query.bind_const_int64(13);
    query.bind_const_int64(42);
    std::chrono::seconds request_timeout(3);
    request_operation operation{ io, query, connection_pool, request_timeout };
    operation.run();

    try
    {
        io.run();
    }
    catch (const std::exception& e)
    {
        std::cout << e.what() << '\n';
    }
}
