#ifndef APQ_CONNECTION_POOL_HPP
#define APQ_CONNECTION_POOL_HPP

#include <utility>
#include <boost/noncopyable.hpp>
#include <boost/make_shared.hpp>
#include <boost/bind/protect.hpp>
#include <yplatform/task_context.h>
#include <apq/detail/connection_pool_service.hpp>
#include <apq/result_format.hpp>
#include <apq/detail/handler_stat.hpp>

namespace apq {

typedef detail::connection_pool_token pool_token;

class connection_pool : private boost::noncopyable
{
public:
    // The type of the service that will be used to provide operations.
    typedef connection_service service_type;
    typedef detail::connection_pool_impl::stat_window_size_t stat_window_size_t;

    explicit connection_pool(
        boost::asio::io_service& ios,
        stat_window_size_t stat_window_size = 10000)
        : service_(boost::asio::use_service<service_type>(ios))
        , impl_(boost::make_shared<detail::connection_pool_impl>(ios, stat_window_size))
    {
    }

    boost::asio::io_service& get_io_service()
    {
        return service_.get_io_service();
    }

    void set_conninfo(const std::string& conninfo)
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        impl_->conninfo_ = conninfo;
    }

    void set_limit(std::size_t count)
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        impl_->max_conn_ = count;
    }

    void set_connect_timeout(time_traits::duration_type timeout)
    {
        impl_->connect_timeout_ = timeout;
    }

    void set_idle_timeout(time_traits::duration_type timeout)
    {
        impl_->idle_timeout_ = timeout;
    }

    void set_queue_timeout(time_traits::duration_type timeout)
    {
        impl_->queue_.timeout(timeout);
    }

    void set_max_cancel_duration(time_traits::duration_type t)
    {
        impl_->max_cancel_duration = t;
    }

    void set_async_resolve(bool enabled)
    {
        impl_->async_resolve_ = enabled;
    }

    void set_ipv6_only(bool enabled)
    {
        impl_->ipv6_only_ = enabled;
    }

    std::size_t free_connections() const
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        return impl_->free_.size();
    }

    std::size_t busy_connections() const
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        return impl_->busy_.size();
    }

    std::size_t pending_connections() const
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        return impl_->temp_;
    }

    std::size_t max_connections() const
    {
        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        return impl_->max_conn_;
    }

    time_traits::duration_type queue_timeout() const
    {
        return impl_->queue_.timeout();
    }

    std::size_t queue_size() const
    {
        return impl_->queue_.size();
    }

    time_traits::duration_type max_cancel_duration() const
    {
        return impl_->max_cancel_duration;
    }

    void clear()
    {
        typedef detail::connection_pool_impl::container_type conn_set;
        conn_set temp_set;

        {
            boost::mutex::scoped_lock lock(impl_->conn_mutex_);
            impl_->busy_.swap(temp_set);
            impl_->free_.clear();
        }

        for (conn_set::iterator it = temp_set.begin(); it != temp_set.end(); ++it)
        {
            service_type::implementation_type conn = *it;
            service_.destroy(conn);
        }

        impl_->queue_.clear();
    }

    typedef boost::function<void(apq::result, apq::row_iterator)> request_handler_t;

    typedef boost::function<void(apq::result, apq::cursor)> request_single_handler_t;

    typedef boost::function<void(apq::result, int)> update_handler_t;

    typedef boost::function<void(apq::result)> execute_handler_t;

    void async_request(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(boost::protect(boost::bind(
            detail::pool_request<request_handler_t>,
            ctx,
            _1,
            _2,
            impl_,
            boost::shared_ptr<pool_token>(),
            boost::cref(query),
            track_latency(std::move(handler)),
            tm,
            rf)));
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        async_request_single(ctx, query, wrap_request_handler(std::move(handler)), rf, tm);
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_single_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(boost::protect(boost::bind(
            detail::pool_request_single<request_single_handler_t>,
            ctx,
            _1,
            _2,
            impl_,
            boost::shared_ptr<pool_token>(),
            boost::cref(query),
            std::move(handler),
            tm,
            rf)));
    }

    void async_update(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        update_handler_t handler,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(boost::protect(boost::bind(
            detail::pool_update<update_handler_t>,
            ctx,
            _1,
            _2,
            impl_,
            boost::shared_ptr<pool_token>(),
            boost::cref(query),
            track_latency(std::move(handler)),
            tm)));
    }

    void async_execute(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        execute_handler_t handler,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(boost::protect(boost::bind(
            detail::pool_execute<execute_handler_t>,
            ctx,
            _1,
            _2,
            impl_,
            boost::shared_ptr<pool_token>(),
            boost::cref(query),
            track_latency(std::move(handler)),
            tm)));
    }

    class token
    {
    public:
        // Cannot use make_shared here, connection_pool_token's constructor
        // is private.
        explicit token(apq::connection_pool& pool)
            : impl_(new detail::connection_pool_token(pool.impl_))
        {
        }

    private:
        boost::shared_ptr<detail::connection_pool_token> impl_;

        friend class connection_pool;
    };

    void async_request(
        yplatform::task_context_ptr ctx,
        const token& t,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(
            t.impl_,
            boost::protect(boost::bind(
                detail::pool_request<request_handler_t>,
                ctx,
                _1,
                _2,
                impl_,
                t.impl_,
                boost::cref(query),
                track_latency(std::move(handler)),
                tm,
                rf)));
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const token& t,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        async_request_single(ctx, t, query, wrap_request_handler(std::move(handler)), rf, tm);
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const token& t,
        const apq::query& query,
        request_single_handler_t handler,
        result_format rf = result_format_text,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(
            t.impl_,
            boost::protect(boost::bind(
                detail::pool_request_single<request_single_handler_t>,
                ctx,
                _1,
                _2,
                impl_,
                t.impl_,
                boost::cref(query),
                std::move(handler),
                tm,
                rf)));
    }

    void async_update(
        yplatform::task_context_ptr ctx,
        const token& t,
        const apq::query& query,
        update_handler_t handler,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(
            t.impl_,
            boost::protect(boost::bind(
                detail::pool_update<update_handler_t>,
                ctx,
                _1,
                _2,
                impl_,
                t.impl_,
                boost::cref(query),
                track_latency(std::move(handler)),
                tm)));
    }

    void async_execute(
        yplatform::task_context_ptr ctx,
        const token& t,
        const apq::query& query,
        execute_handler_t handler,
        time_traits::duration_type tm = time_traits::pos_infin_duration())
    {
        alloc(
            t.impl_,
            boost::protect(boost::bind(
                detail::pool_execute<execute_handler_t>,
                ctx,
                _1,
                _2,
                impl_,
                t.impl_,
                boost::cref(query),
                track_latency(std::move(handler)),
                tm)));
    }

    struct stats
    {
        typedef detail::connection_pool_impl::dropped_connections_stat dropped_connections_stat;

        std::size_t free_connections = 0;
        std::size_t busy_connections = 0;
        std::size_t pending_connections = 0;
        std::size_t max_connections = 0;
        std::size_t queue_size = 0;

        unsigned long average_request_roundtrip_usec = 0;
        unsigned long average_request_db_latency_usec = 0;
        unsigned long average_wait_time_usec = 0;

        dropped_connections_stat num_dropped_connections;
    };

    stats get_stats() const
    {
        stats result;
        {
            const boost::mutex::scoped_lock lock(impl_->conn_mutex_);
            result.free_connections = impl_->free_.size();
            result.busy_connections = impl_->busy_.size();
            result.pending_connections = impl_->temp_;
            result.max_connections = impl_->max_conn_;

            result.average_request_roundtrip_usec = static_cast<unsigned long>(
                time_traits::to_usec(impl_->average_roundtrip_time_->compute()));
            result.average_request_db_latency_usec = static_cast<unsigned long>(
                time_traits::to_usec(impl_->average_db_latency_->compute()));
            result.average_wait_time_usec = static_cast<unsigned long>(
                time_traits::to_usec(impl_->average_wait_time_->compute()));

            result.num_dropped_connections = impl_->num_dropped_connections_;
        }
        result.queue_size = queue_size();
        return result;
    }

private:
    request_single_handler_t wrap_request_handler(request_handler_t handler) const
    {
        return [handler = std::move(handler)](apq::result r, apq::cursor c) {
            if (auto ctx = c.get())
            {
                handler(r, std::move(row(*ctx)));
                continuation(*ctx).resume();
            }
            else
            {
                handler(r, {});
            }
        };
    }

    template <typename Handler>
    Handler track_latency(Handler handler) const
    {
        return detail::track_latency(std::move(handler), impl_->average_roundtrip_time_);
    }

    template <typename Handler>
    void alloc(boost::shared_ptr<pool_token> token, Handler handler)
    {
        if (!token)
        {
            handler(
                make_error_code(boost::asio::error::not_found),
                boost::shared_ptr<detail::connection_impl>());
            return;
        }

        boost::mutex::scoped_lock lock(impl_->conn_mutex_);
        detail::connection_pool_impl::captured_container_type::iterator it =
            impl_->captured_.find(reinterpret_cast<std::ptrdiff_t>(token.get()));
        if (it != impl_->captured_.end())
        {
            lock.unlock();
            handler(boost::system::error_code(), it->second);
            return;
        }
        lock.unlock();

        alloc(handler);
    }

    template <typename Handler>
    void alloc(Handler handler)
    {
        for (;;)
        {
            boost::mutex::scoped_lock lock(impl_->conn_mutex_);

            if (impl_->free_.empty())
            {
                if (count() >= impl_->max_conn_)
                {
                    lock.unlock();

                    // Enqueue request.
                    impl_->queue_.push(handler);
                    return;
                }
                else
                {
                    ++impl_->temp_;
                    impl_->average_wait_time_->add_zero();

                    lock.unlock();

                    // Open a new connection.
                    typedef detail::pool_connect_op<detail::connection_pool_impl, Handler> op;
                    boost::make_shared<op>(service_, impl_, handler)->perform();
                    return;
                }
            }
            else
            {
                // Use an existing connection.
                detail::connection_pool_impl::container_type::iterator it = impl_->free_.begin();
                service_type::implementation_type conn = *it;
                impl_->free_.erase(it);

                // If something is wrong, drop the conection and try again.
                if (conn->close_time_ <= apq::time_traits::now())
                {
                    ++impl_->num_dropped_connections_.timed_out;
                    continue;
                }
                if (PQstatus(conn->conn_.get()) != CONNECTION_OK)
                {
                    ++impl_->num_dropped_connections_.failed;
                    continue;
                }
                if (PQisBusy(conn->conn_.get()))
                {
                    ++impl_->num_dropped_connections_.busy;
                    continue;
                }
                if (PQgetResult(conn->conn_.get()) != NULL)
                {
                    ++impl_->num_dropped_connections_.with_result;
                    continue;
                }

                impl_->busy_.insert(conn);
                impl_->average_wait_time_->add_zero();

                lock.unlock();

                service_.get_io_service().post(
                    boost::bind(handler, boost::system::error_code(), conn));
                return;
            }
        }
    }

    std::size_t count() const
    {
        return impl_->free_.size() + impl_->busy_.size() + impl_->temp_;
    }

    service_type& service_;

    boost::shared_ptr<detail::connection_pool_impl> impl_;
};

class connection_pool_holder
{
public:
    explicit connection_pool_holder(apq::connection_pool& pool) : pool_(pool), token_(pool)
    {
    }

    apq::connection_pool& next_layer()
    {
        return pool_;
    }

    typedef apq::connection_pool::request_handler_t request_handler_t;
    typedef apq::connection_pool::request_single_handler_t request_single_handler_t;
    typedef apq::connection_pool::update_handler_t update_handler_t;
    typedef apq::connection_pool::execute_handler_t execute_handler_t;
    typedef apq::time_traits::duration_type time_duration;

    void async_request(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_duration tm = time_traits::pos_infin_duration())
    {
        pool_.async_request(ctx, token_, query, std::move(handler), rf, tm);
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_handler_t handler,
        result_format rf = result_format_text,
        time_duration tm = time_traits::pos_infin_duration())
    {
        pool_.async_request_single(ctx, token_, query, std::move(handler), rf, tm);
    }

    void async_request_single(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        request_single_handler_t handler,
        result_format rf = result_format_text,
        time_duration tm = time_traits::pos_infin_duration())
    {
        pool_.async_request_single(ctx, token_, query, std::move(handler), rf, tm);
    }

    void async_update(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        update_handler_t handler,
        time_duration tm = time_traits::pos_infin_duration())
    {
        pool_.async_update(ctx, token_, query, std::move(handler), tm);
    }

    void async_execute(
        yplatform::task_context_ptr ctx,
        const apq::query& query,
        execute_handler_t handler,
        time_duration tm = time_traits::pos_infin_duration())
    {
        pool_.async_execute(ctx, token_, query, std::move(handler), tm);
    }

private:
    apq::connection_pool& pool_;
    apq::connection_pool::token token_;
};

inline yplatform::task_context_ptr fake_task_context()
{
    // Note. To minimize the potential of data races on shared fakes,
    // `task_context::fake` is not used; a distinct instance is created and used instead.
    static const auto instance = boost::make_shared<yplatform::task_context>("fake-apq-ctx");
    return instance;
}

} // namespace apq

#endif
