#ifndef APQ_DETAIL_BIND_HPP
#define APQ_DETAIL_BIND_HPP

#include <cstring>
#include <string>
#include <vector>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <boost/asio.hpp>
#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_io.hpp>
#include <libpq-fe.h>
#include <apq/detail/streamer.hpp>
#include <apq/detail/hton.hpp>

namespace apq { namespace detail {

struct bind_value_base
{
    virtual ~bind_value_base()
    {
    }
    virtual std::ostream& stream(std::ostream&) const = 0;
};

template <typename T>
struct basic_bind_value : public bind_value_base
{
    virtual const T& value() const = 0;
    virtual std::ostream& stream(std::ostream& s) const
    {
        typedef typename boost::remove_const<T>::type value_type;
        return streamer<value_type>::stream(s, value());
    }
};

enum class bind_value_type
{
    INT64,
    DOUBLE,
    STRING,
    INT64_ARRAY,
    UINT64_ARRAY,
    STRING_ARRAY,
    BYTE_ARRAY,
    PGNULL,
    UUID
};

typedef std::pair<bind_value_type, boost::shared_ptr<detail::bind_value_base>> bind_pair_t;

inline bind_value_type param_type(const bind_pair_t& v)
{
    return v.first;
}

template <typename T>
inline const T& param_value(const bind_pair_t& v)
{
    return boost::dynamic_pointer_cast<detail::basic_bind_value<T>>(v.second)->value();
}

template <typename T>
struct cref_bind_value : public basic_bind_value<T>
{
    explicit cref_bind_value(const T& value) : value_(value)
    {
    }
    const T& value() const
    {
        return value_;
    }
    const T& value_;
};

template <typename T>
struct const_bind_value : public basic_bind_value<T>
{
    explicit const_bind_value(const T& value) : value_(value)
    {
    }
    explicit const_bind_value(T&& value) : value_(std::move(value))
    {
    }
    const T& value() const
    {
        return value_;
    }
    const T value_;
};

struct const_bytes_value : public bind_value_base
{
    const_bytes_value(const void* ptr, int sz) : ptr_(ptr), sz_(sz)
    {
    }
    virtual std::ostream& stream(std::ostream& s) const
    {
        return streamer<void*>::stream(s, ptr_, sz_);
    }
    const void* ptr_;
    int sz_;
};

template <typename T>
boost::shared_ptr<bind_value_base> make_cref_bind_value(const T& value)
{
    return boost::make_shared<cref_bind_value<T>>(value);
}

template <typename T>
boost::shared_ptr<bind_value_base> make_const_bind_value(T&& value)
{
    return boost::make_shared<const_bind_value<typename std::decay<T>::type>>(
        std::forward<T>(value));
}

inline boost::shared_ptr<bind_value_base> make_const_bytes_value(const void* ptr, int sz)
{
    return boost::make_shared<const_bytes_value>(ptr, sz);
}

enum pq_internal_types
{
    int8oid = 20,
    int8arrayoid = 1016,
    float8oid = 701,
    varcharoid = 1043,
    varchararrayoid = 1015,
    byteaoid = 17,
    uuidoid = 2950
};

template <Oid OID, Oid ArrayOID, int Format = 1>
struct traits_base
{
    static const Oid oid = OID;
    static const Oid array_oid = ArrayOID;
    static const int format = Format; // Binary / text.
};

template <typename T>
struct traits : public traits_base<InvalidOid, InvalidOid>
{
};

template <>
struct traits<int64_t> : public traits_base<20, 1016>
{
};

template <>
struct traits<std::string> : public traits_base<1043, 1015, 0>
{
};

struct bind_parameters
{
    std::vector<Oid> types;
    std::vector<const char*> values;
    std::vector<int> lengths;
    std::vector<int> formats;
    std::vector<boost::shared_ptr<void>> buffers;
    std::vector<std::string> strings;

    bind_parameters(std::size_t size)
    {
        types.reserve(size);
        values.reserve(size);
        lengths.reserve(size);
        formats.reserve(size);
        buffers.reserve(size);
        strings.reserve(size);
    }
};

inline void bind_string(const std::string& v, bind_parameters& params)
{
    params.types.push_back(0); // Untyped literal string
    params.values.push_back(v.c_str());
    params.lengths.push_back(0); // Ignored for text format
    params.formats.push_back(0); // Text format
}

inline void bind_vector(const std::vector<std::string>& v, bind_parameters& params)
{
    // Header size: dims + has_null + oid + (dim_length + dim_start).
    std::size_t data_sz = 4 + 4 + 4 + 8;

    // Body size: element_sz + element
    for (std::vector<std::string>::const_iterator it = v.begin(); it != v.end(); ++it)
    {
        data_sz += (4 + it->length());
    }

    boost::shared_ptr<void> data(operator new(data_sz), std::ptr_fun(operator delete));
    uint32_t* data_ptr = static_cast<uint32_t*>(data.get());

    *data_ptr++ = htonl(1);                               // One dimension
    *data_ptr++ = htonl(0);                               // No nulls
    *data_ptr++ = htonl(varcharoid);                      // Element type
    *data_ptr++ = htonl(static_cast<uint32_t>(v.size())); // Dimension length
    *data_ptr++ = htonl(1);                               // Dimension start index

    for (std::vector<std::string>::const_iterator it = v.begin(); it != v.end(); ++it)
    {
        // Element size.
        *data_ptr++ = htonl(static_cast<uint32_t>(it->length()));

        // Element itself.
        char* dest = static_cast<char*>(static_cast<void*>(data_ptr));
        std::strncpy(dest, it->c_str(), it->length());
        dest += it->length();
        data_ptr = static_cast<uint32_t*>(static_cast<void*>(dest));
    }

    assert(static_cast<void*>(data_ptr) == static_cast<char*>(data.get()) + data_sz);

    params.types.push_back(varchararrayoid);
    params.values.push_back(static_cast<const char*>(data.get()));
    params.lengths.push_back(static_cast<int>(data_sz));
    params.formats.push_back(1);
    params.buffers.push_back(data);
}

inline void bind_vector(const std::vector<int64_t>& v, bind_parameters& params)
{
    // Header size: dims + has_null + oid + (dim_length + dim_start).
    std::size_t data_sz = 4 + 4 + 4 + 8;

    // Body size: element_sz + element
    data_sz += (4 + sizeof(int64_t)) * v.size();

    boost::shared_ptr<void> data(operator new(data_sz), std::ptr_fun(operator delete));
    uint32_t* data_ptr = static_cast<uint32_t*>(data.get());

    *data_ptr++ = htonl(1);                               // One dimension
    *data_ptr++ = htonl(0);                               // No nulls
    *data_ptr++ = htonl(int8oid);                         // Element type
    *data_ptr++ = htonl(static_cast<uint32_t>(v.size())); // Dimension length
    *data_ptr++ = htonl(1);                               // Dimension start index

    for (std::vector<int64_t>::const_iterator it = v.begin(); it != v.end(); ++it)
    {
        // Element size.
        *data_ptr++ = htonl(sizeof(int64_t));

        // Element itself.
        uint64_t* dest = static_cast<uint64_t*>(static_cast<void*>(data_ptr));
        *dest++ = htonll(static_cast<uint64_t>(*it));
        data_ptr = static_cast<uint32_t*>(static_cast<void*>(dest));
    }

    assert(static_cast<void*>(data_ptr) == static_cast<char*>(data.get()) + data_sz);

    params.types.push_back(int8arrayoid);
    params.values.push_back(static_cast<const char*>(data.get()));
    params.lengths.push_back(static_cast<int>(data_sz));
    params.formats.push_back(1);
    params.buffers.push_back(data);
}

inline void bind_vector(const std::vector<uint64_t>& v, bind_parameters& params)
{
    // Header size: dims + has_null + oid + (dim_length + dim_start).
    std::size_t data_sz = 4 + 4 + 4 + 8;

    // Body size: element_sz + element
    data_sz += (4 + sizeof(uint64_t)) * v.size();

    boost::shared_ptr<void> data(operator new(data_sz), std::ptr_fun(operator delete));
    uint32_t* data_ptr = static_cast<uint32_t*>(data.get());

    *data_ptr++ = htonl(1);                               // One dimension
    *data_ptr++ = htonl(0);                               // No nulls
    *data_ptr++ = htonl(int8oid);                         // Element type
    *data_ptr++ = htonl(static_cast<uint32_t>(v.size())); // Dimension length
    *data_ptr++ = htonl(1);                               // Dimension start index

    for (std::vector<uint64_t>::const_iterator it = v.begin(); it != v.end(); ++it)
    {
        // Element size.
        *data_ptr++ = htonl(sizeof(uint64_t));

        // Element itself.
        uint64_t* dest = static_cast<uint64_t*>(static_cast<void*>(data_ptr));
        *dest++ = htonll(*it);
        data_ptr = static_cast<uint32_t*>(static_cast<void*>(dest));
    }

    assert(static_cast<void*>(data_ptr) == static_cast<char*>(data.get()) + data_sz);

    params.types.push_back(int8arrayoid);
    params.values.push_back(static_cast<const char*>(data.get()));
    params.lengths.push_back(static_cast<int>(data_sz));
    params.formats.push_back(1);
    params.buffers.push_back(data);
}

inline void bind_int(int64_t v, bind_parameters& params)
{
    params.strings.emplace_back(std::to_string(v));
    bind_string(params.strings.back(), params);
}

inline void bind_double(double v, bind_parameters& params)
{
    params.strings.emplace_back(std::to_string(v));
    bind_string(params.strings.back(), params);
}

inline void bind_bytes(const const_bytes_value& v, bind_parameters& params)
{
    params.types.push_back(byteaoid);
    params.values.push_back(static_cast<const char*>(v.ptr_));
    params.lengths.push_back(v.sz_);
    params.formats.push_back(1);
}

inline void bind_uuid(const boost::uuids::uuid uuid, bind_parameters& params)
{
    boost::shared_ptr<void> data(operator new(sizeof(uuid)), std::ptr_fun(operator delete));
    auto* ptr = static_cast<char*>(data.get());
    std::copy_n(uuid.begin(), sizeof(uuid), ptr);

    params.types.push_back(uuidoid);
    params.values.push_back(ptr);
    params.lengths.push_back(sizeof(uuid));
    params.formats.push_back(1);
    params.buffers.push_back(std::move(data));
}

inline void bind_value(const bind_pair_t& v, bind_parameters& params)
{
    switch (param_type(v))
    {
    case bind_value_type::INT64:
        bind_int(param_value<int64_t>(v), params);
        break;

    case bind_value_type::INT64_ARRAY:
        bind_vector(param_value<std::vector<int64_t>>(v), params);
        break;

    case bind_value_type::UINT64_ARRAY:
        bind_vector(param_value<std::vector<uint64_t>>(v), params);
        break;

    case bind_value_type::DOUBLE:
        bind_double(param_value<double>(v), params);
        break;

    case bind_value_type::STRING:
        bind_string(param_value<std::string>(v), params);
        break;

    case bind_value_type::STRING_ARRAY:
        bind_vector(param_value<std::vector<std::string>>(v), params);
        break;

    case bind_value_type::BYTE_ARRAY:
        bind_bytes(*(boost::static_pointer_cast<const_bytes_value>(v.second).get()), params);
        break;

    case bind_value_type::PGNULL:
        params.values.push_back(NULL);
        params.types.push_back(0);
        params.lengths.push_back(0);
        params.formats.push_back(0);
        break;

    case bind_value_type::UUID:
        bind_uuid(param_value<boost::uuids::uuid>(v), params);
        break;
    }
}

inline bind_parameters make_bind_parameters(const std::vector<bind_pair_t>& values)
{
    bind_parameters params(values.size());
    for (const auto& i : values)
    {
        bind_value(i, params);
    }
    return params;
}

} // namespace detail
} // namespace apq

#endif
