#ifndef APQ_DETAIL_BIND_RESULT_HPP
#define APQ_DETAIL_BIND_RESULT_HPP

#include <string>
#include <sstream>
#include <boost/utility/enable_if.hpp>
#include <boost/type_traits/is_integral.hpp>
#include <boost/type_traits/is_floating_point.hpp>
#include <boost/mpl/has_xxx.hpp>
#include <boost/mpl/and.hpp>
#include <boost/mpl/not.hpp>
#include <boost/fusion/algorithm/iteration/for_each.hpp>
#include <boost/fusion/support/is_sequence.hpp>
#include <boost/fusion/sequence/intrinsic/size.hpp>
#include <boost/optional.hpp>
#include <boost/uuid/uuid.hpp>
#include <libpq-fe.h>
#include <apq/detail/hton.hpp>

namespace apq { namespace detail {

BOOST_MPL_HAS_XXX_TRAIT_DEF(iterator)
BOOST_MPL_HAS_XXX_TRAIT_DEF(mapped_type)

template <typename Binder>
class composite_adaptor
{
public:
    explicit composite_adaptor(const Binder& binder) : binder_(binder)
    {
    }

    template <typename T>
    void operator()(T& t) const
    {
        // Read element OID.
        Oid oid;
        binder_(oid, sizeof(oid));

        // Read element size.
        int32_t size;
        binder_(size, sizeof(size));

        // Read element itself if it is non NULL.
        if (size > -1) binder_(t, size);
    }

private:
    const Binder& binder_;
};

class result_binder
{
public:
    result_binder(const PGresult* result, int row, int col)
        : data_(PQgetvalue(result, row, col))
        , size_(PQgetlength(result, row, col))
        , data_end_(data_ + size_)
        , is_null_(PQgetisnull(result, row, col))
        , column_(PQfname(result, col))
    {
    }

    template <typename T>
    void operator()(T& t) const
    {
        if (!is_null_) read(t, size_);
        if (data_ != data_end_)
            throw std::logic_error("Some data left in the buffer. Run! Get to da choppa!");
    }

    template <typename T>
    void operator()(T& t, int size) const
    {
        read(t, size);
    }

private:
    template <typename T>
    void read(T& t, int size, typename boost::enable_if<boost::is_integral<T>>::type* = 0) const
    {
        check_size(size);
        if (sizeof(t) < static_cast<unsigned int>(size))
        {
            std::ostringstream ss;
            ss << "Integer value is too big: " << size << ", expected " << sizeof(t)
               << " at column " << column_;
            throw std::overflow_error(ss.str());
        }
// Here we push old-style-cast warning due to the system macroses are used
// under the -O2
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wold-style-cast"

#ifdef __GNUC_PREREQ
#if __GNUC_PREREQ(4, 9)
#pragma GCC diagnostic ignored "-Wextra"
#endif
#endif
        if (size == 1)
            t = static_cast<T>(*static_cast<const int8_t*>(static_cast<const void*>(data_)));
        else if (size == 2)
            t = static_cast<T>(
                ntohs(*static_cast<const uint16_t*>(static_cast<const void*>(data_))));
        else if (size == 4)
            t = static_cast<T>(
                ntohl(*static_cast<const uint32_t*>(static_cast<const void*>(data_))));
        else if (size == 8)
            t = static_cast<T>(
                ntohll(*static_cast<const uint64_t*>(static_cast<const void*>(data_))));
        data_ += size;
#ifdef __GNUC_PREREQ
#if __GNUC_PREREQ(4, 9)
#pragma GCC diagnostic pop
#endif
#endif

#pragma GCC diagnostic pop
    }

    template <typename T>
    void read(T& t, int size, typename boost::enable_if<boost::is_floating_point<T>>::type* = 0)
        const
    {
        check_size(size);
        if (sizeof(t) < static_cast<unsigned int>(size))
        {
            std::ostringstream ss;
            ss << "Floating point value is too big: " << size << ", expected " << sizeof(t)
               << " at column " << column_;
            throw std::overflow_error(ss.str());
        }
        if (size == 4) t = *static_cast<const float*>(static_cast<const void*>(data_));
        else if (size == 8)
            t = *static_cast<const double*>(static_cast<const void*>(data_));
        data_ += size;
    }

    template <typename Traits, typename Allocator>
    void read(std::basic_string<char, Traits, Allocator>& t, int size) const
    {
        t.assign(data_, static_cast<size_t>(size));
        data_ += size;
    }

    template <typename T, std::size_t N>
    void read(std::array<T, N>& t, int) const
    {
        read_array<T>(t.begin());
    }

    template <typename T, std::size_t N>
    void read(boost::array<T, N>& t, int) const
    {
        read_array<T>(t.begin());
    }

    template <typename T>
    void read(
        T& t,
        int,
        typename boost::enable_if<
            boost::mpl::and_<has_iterator<T>, boost::mpl::not_<has_mapped_type<T>>>>::type* =
            0) const
    {
        typedef typename std::iterator_traits<typename T::iterator>::value_type value_type;
        read_array<value_type>(std::inserter(t, t.end()));
    }

    template <typename T, typename OutputIterator>
    void read_array(OutputIterator dest) const
    {
        // Read dimensions count.
        int32_t dimensions;
        (*this)(dimensions, sizeof(dimensions));
        if (dimensions > 1) throw std::logic_error("No support for multidimensional arrays");

        // Read nulls flag.
        int32_t have_nulls;
        (*this)(have_nulls, sizeof(have_nulls));

        // Read elements OID.
        Oid oid;
        (*this)(oid, sizeof(oid));

        if (dimensions > 0)
        {
            // Read dimension length.
            int32_t length;
            (*this)(length, sizeof(length));

            // Read dimension start index.
            int32_t start_index;
            (*this)(start_index, sizeof(start_index));

            if (length == 0) return;

            for (; length; --length)
            {
                // Read element size.
                int32_t value_size;
                (*this)(value_size, sizeof(value_size));

                // Read value if it is not NULL.
                T value;
                if (value_size > -1) (*this)(value, value_size);
                *dest++ = value;
            }
        }
    }

    template <typename T>
    void read(
        T& t,
        int,
        typename boost::enable_if<boost::fusion::traits::is_sequence<T>>::type* = 0) const
    {
        // Read columns count.
        int32_t cols = 0;
        (*this)(cols, sizeof(cols));
        if (boost::fusion::size(t) != static_cast<int>(cols))
        {
            std::ostringstream ss;
            ss << "Composites sizes do not match: " << cols << ", expected "
               << boost::fusion::size(t) << " columns at column " << column_;
            throw std::logic_error(ss.str());
        }

        // Read columns values.
        boost::fusion::for_each(t, composite_adaptor<result_binder>(*this));
    }

    template <typename T>
    void read(boost::optional<T>& t, int size) const
    {
        t = T();
        read(t.get(), size);
    }

    void read(boost::uuids::uuid& t, int size) const
    {
        check_size(size);
        if (sizeof(t) < size_t(size))
        {
            std::ostringstream ss;
            ss << "UUID value is too big: " << size << ", expected " << sizeof(t) << " at column "
               << column_;
            throw std::overflow_error(ss.str());
        }

        std::copy_n(data_, sizeof(t), t.begin());
        data_ += size;
    }

    void check_size(int size) const
    {
        if (data_end_ - data_ < size)
        {
            std::ostringstream ss;
            ss << "No enough data left. Trying to read " << size << "B, left "
               << (data_end_ - data_) << "B at column " << column_;
            throw std::logic_error(ss.str());
        }
    }

    mutable const char* data_;
    const int size_;
    const char* const data_end_;
    const bool is_null_;
    const char* const column_;
};

template <typename T>
inline void bind_result(T& t, const PGresult* result, int row, int col)
{
    result_binder binder(result, row, col);
    binder(t);
}

} // namespace detail
} // namespace apq

#endif
