#pragma once

#include <apq/detail/connection_info.hpp>
#include <apq/result.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/make_shared.hpp>
#include <libpq-fe.h>

namespace apq::detail {

template <typename Handler, typename Connection>
struct connect_op
{
    connect_op(boost::shared_ptr<Connection>& impl, Handler handler)
        : impl_(impl), handler_(std::move(handler))
    {
    }

    void done(boost::system::error_code err = {}, const std::string& reason = {})
    {
        impl_->strand_->post(std::bind(std::move(handler_), result(err, reason)));
    }

    void perform(const connection_info& conninfo)
    {
        try
        {
            impl_->start_connect(conninfo);
            if (!impl_->is_connected())
            {
                return done(error::network, "Failed to allocate a new PGconn");
            }

            if (impl_->get_status() == CONNECTION_BAD)
            {
                std::ostringstream msg;
                msg << "Connection is bad";
                const char* error = impl_->get_error_message();
                if (error != nullptr)
                    msg << ": " << boost::trim_right_copy(boost::as_literal(error));
                if (conninfo.size())
                {
                    auto conninfo_str = boost::join(
                        conninfo | boost::adaptors::transformed([](auto&& param) {
                            return param.first + "=" + param.second;
                        }),
                        " ");
                    msg << " (" << conninfo_str << ")";
                }
                return done(error::network, msg.str());
            }

            impl_->init_multihost();
            const auto [ec, reason] = impl_->assign_socket();
            if (ec)
            {
                return done(ec, reason);
            }

            return write_poll(impl_, impl_->strand_->wrap(*this));
        }
        catch (const std::exception& e)
        {
            done(error::network, e.what());
        }
        catch (...)
        {
            done(error::network, std::string("unknown exception in ") + __PRETTY_FUNCTION__);
        }
    }

    void operator()(boost::system::error_code ec, std::size_t = 0)
    {
        if (impl_->is_multihost_)
        {
            // Make sure that we won't get stuck in a blocking read or write
            // inside PQconnectPoll, then give libpq a chance to try another host.
            // If there is an error, we'll get it from PQconnectPoll.
            if (ec == boost::asio::error::operation_aborted)
            {
                shutdown(impl_);
            }
        }
        else if (ec)
        {
            return done(ec);
        }

        try
        {
            auto poll_result = impl_->connection_poll();
            if (poll_result != PGRES_POLLING_FAILED)
            {
                const auto [ec, reason] = impl_->refresh_socket();
                if (ec)
                {
                    return done(ec, reason);
                }
            }
            switch (poll_result)
            {
            case PGRES_POLLING_OK:
                return done();

            case PGRES_POLLING_WRITING:
                return write_poll(impl_, impl_->strand_->wrap(*this));

            case PGRES_POLLING_READING:
                return read_poll(impl_, impl_->strand_->wrap(*this));

            case PGRES_POLLING_FAILED:
            default:
                break;
            }
        }
        catch (...)
        {
        }

        done(error::network, error::make_error_msg("PQconnectPoll", impl_->get_error_message()));
    }

    boost::shared_ptr<Connection> impl_;
    Handler handler_;
};

}