#ifndef APQ_DETAIL_CONNECTION_POOL_SERVICE_HPP
#define APQ_DETAIL_CONNECTION_POOL_SERVICE_HPP

#include <cassert>
#include <algorithm>
#include <set>
#include <deque>
#include <boost/thread/mutex.hpp>
#include <boost/function.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <apq/connection_service.hpp>
#include <apq/result_format.hpp>
#include <apq/time_traits.hpp>
#include <apq/detail/moving_average_duration.hpp>
#include <apq/detail/pool_connect_op.hpp>

namespace apq {

class connection_pool;

namespace detail {

struct expiring_request
{
    typedef boost::function<void(boost::system::error_code, boost::shared_ptr<connection_impl>)>
        handler_t;

    expiring_request(handler_t handler, apq::time_traits::duration_type t)
        : handler_(handler)
        , expiry_time_(apq::time_traits::add(apq::time_traits::now(), t))
        , enqueue_time_(time_traits::now())
    {
    }

    handler_t handler_;
    apq::time_traits::time_type expiry_time_;
    time_traits::time_type enqueue_time_;
};

class expiring_request_queue
{
public:
    expiring_request_queue(
        boost::asio::io_service& ios,
        const moving_average_duration_ptr& average_wait_time)
        : timer_(ios)
        , timeout_(apq::time_traits::pos_infin_duration())
        , average_wait_time_(average_wait_time)
    {
        timer_.expires_at(apq::time_traits::pos_infin());
    }

    boost::asio::io_service& get_io_service()
    {
        return timer_.get_io_service();
    }

    std::size_t size() const
    {
        boost::mutex::scoped_lock lock(mutex_);
        return queue_.size();
    }

    apq::time_traits::duration_type timeout() const
    {
        boost::mutex::scoped_lock lock(mutex_);
        return timeout_;
    }

    void timeout(apq::time_traits::duration_type timeout)
    {
        boost::mutex::scoped_lock lock(mutex_);
        timeout_ = timeout;
    }

    void push(const expiring_request::handler_t& handler)
    {
        boost::mutex::scoped_lock lock(mutex_);
        queue_.push_back(expiring_request(handler, timeout_));

        if (timer_.expires_at() == apq::time_traits::pos_infin())
        {
            timer_.expires_from_now(timeout_);
            timer_.async_wait(boost::bind(&expiring_request_queue::handle_timer, this, _1));
        }
    }

    bool pop(const boost::shared_ptr<connection_impl>& conn)
    {
        boost::mutex::scoped_lock lock(mutex_);

        if (queue_.empty()) return false;

        if (average_wait_time_)
        {
            average_wait_time_->add_value(time_traits::now() - queue_.front().enqueue_time_);
        }
        expiring_request::handler_t handler = queue_.front().handler_;
        queue_.pop_front();

        if (queue_.empty())
        {
            timer_.cancel();
            timer_.expires_at(apq::time_traits::pos_infin());
        }
        else
        {
            timer_.expires_at(queue_.front().expiry_time_);
            timer_.async_wait(boost::bind(&expiring_request_queue::handle_timer, this, _1));
        }

        lock.unlock();

        handler_invoke(handler, boost::system::error_code(), conn);

        return true;
    }

    void clear()
    {
        std::deque<expiring_request> temp;
        boost::mutex::scoped_lock lock(mutex_);
        queue_.swap(temp);
        timer_.cancel();
        timer_.expires_at(apq::time_traits::pos_infin());
        lock.unlock();

        for (auto& it : temp)
        {
            handler_invoke(
                it.handler_,
                make_error_code(boost::system::errc::operation_canceled),
                boost::shared_ptr<connection_impl>());
        }
    }

private:
    void handle_timer(const boost::system::error_code& ec)
    {
        if (ec == boost::asio::error::operation_aborted) return;

        for (;;)
        {
            boost::mutex::scoped_lock lock(mutex_);

            if (queue_.empty())
            {
                timer_.expires_at(apq::time_traits::pos_infin());
                return;
            }

            if (queue_.front().expiry_time_ >= apq::time_traits::now())
            {
                timer_.expires_at(queue_.front().expiry_time_);
                timer_.async_wait(boost::bind(&expiring_request_queue::handle_timer, this, _1));
                return;
            }

            if (average_wait_time_)
            {
                average_wait_time_->add_value(time_traits::now() - queue_.front().enqueue_time_);
            }
            expiring_request::handler_t handler = queue_.front().handler_;
            queue_.pop_front();

            lock.unlock();

            handler_invoke(
                handler,
                make_error_code(::apq::error::request_queue_timed_out),
                boost::shared_ptr<connection_impl>());
        }
    }

    void handler_invoke(
        expiring_request::handler_t& handler,
        boost::system::error_code ec,
        boost::shared_ptr<connection_impl> ptr)
    {
        using boost::asio::asio_handler_invoke;
        asio_handler_invoke([&]() mutable { handler(ec, ptr); }, &handler);
    }

    std::deque<expiring_request> queue_;
    apq::time_traits::timer_type timer_;
    apq::time_traits::duration_type timeout_;
    moving_average_duration_ptr average_wait_time_;
    mutable boost::mutex mutex_;
};

struct connection_pool_impl
{
    typedef moving_average_duration_t::window_size_t stat_window_size_t;

    explicit connection_pool_impl(boost::asio::io_service& ios, stat_window_size_t stat_window_size)
        : temp_(0)
        , max_conn_(1)
        , connect_timeout_(apq::time_traits::pos_infin_duration())
        , idle_timeout_(apq::time_traits::pos_infin_duration())
        , average_roundtrip_time_(new moving_average_duration_t(stat_window_size))
        , average_db_latency_(new moving_average_duration_t(stat_window_size))
        , average_wait_time_(new moving_average_duration_t(stat_window_size))
        , num_dropped_connections_(dropped_connections_stat())
        , queue_(ios, average_wait_time_)
        , max_cancel_duration(apq::time_traits::milliseconds(100))
    {
    }

    struct dropped_connections_stat
    {
        uint64_t timed_out = 0;
        uint64_t failed = 0;
        uint64_t busy = 0;
        uint64_t with_result = 0;
    };

    struct less
    {
        template <typename T>
        bool operator()(const T& x, const T& y) const
        {
            return x->conn_.get() > y->conn_.get();
        }
    };

    typedef connection_impl connection_type;
    typedef std::set<boost::shared_ptr<connection_type>, less> container_type;
    typedef std::map<std::ptrdiff_t, boost::shared_ptr<connection_type>> captured_container_type;

    container_type free_;
    container_type busy_;
    captured_container_type captured_;
    std::size_t temp_;
    std::size_t max_conn_;
    apq::time_traits::duration_type connect_timeout_;
    apq::time_traits::duration_type idle_timeout_;
    bool async_resolve_ = false;
    bool ipv6_only_ = false;
    boost::mutex conn_mutex_;

    moving_average_duration_ptr average_roundtrip_time_;
    moving_average_duration_ptr average_db_latency_;
    moving_average_duration_ptr average_wait_time_;
    dropped_connections_stat num_dropped_connections_;

    expiring_request_queue queue_;

    std::string conninfo_;

    apq::time_traits::duration_type max_cancel_duration;
};

inline void pool_free(
    boost::shared_ptr<connection_pool_impl>& impl,
    boost::shared_ptr<connection_impl>& conn,
    bool reuse = true)
{
    try
    {
        // Drop the connection if it's broken, not in idle state,
        // or we don't intend to reuse it.
        if (!reuse || PQstatus(conn->conn_.get()) != CONNECTION_OK ||
            PQtransactionStatus(conn->conn_.get()) != PQTRANS_IDLE)
        {
            boost::mutex::scoped_lock lock(impl->conn_mutex_);
            impl->busy_.erase(conn);
            return;
        }

        // See if we have queued requests and serve one.
        if (impl->queue_.pop(conn)) return;

        // Return connection to the pool.
        boost::mutex::scoped_lock lock(impl->conn_mutex_);
        impl->busy_.erase(conn);
        conn->close_time_ = apq::time_traits::add(apq::time_traits::now(), impl->idle_timeout_);
        impl->free_.insert(conn);
    }
    catch (...)
    {
        boost::mutex::scoped_lock lock(impl->conn_mutex_);
        impl->busy_.erase(conn);
    }
}

class connection_pool_token : private boost::noncopyable
{
public:
    ~connection_pool_token()
    {
        release(true);
    }

    // Forcibly release the captured connection with or without attempting to
    // reuse it. The latter is needed in cases when we want to simply drop the
    // connection.
    void release(bool reuse)
    {
        boost::mutex::scoped_lock lock(pool_->conn_mutex_);
        connection_pool_impl::captured_container_type::iterator it =
            pool_->captured_.find(reinterpret_cast<std::ptrdiff_t>(this));
        if (it != pool_->captured_.end())
        {
            boost::shared_ptr<connection_impl> conn = it->second;
            pool_->captured_.erase(it);
            lock.unlock();
            pool_free(pool_, conn, reuse);
        }
    }

private:
    explicit connection_pool_token(boost::shared_ptr<detail::connection_pool_impl> pool)
        : pool_(pool)
    {
    }

    boost::shared_ptr<detail::connection_pool_impl> pool_;

    friend class apq::connection_pool;
};

struct pooled_connection
{
    pooled_connection(
        boost::shared_ptr<connection_pool_impl> pool,
        boost::shared_ptr<connection_impl> conn)
        : pool_(pool), conn_(conn)
    {
    }

    pooled_connection(
        boost::shared_ptr<connection_pool_impl> pool,
        boost::shared_ptr<connection_pool_token> token,
        boost::shared_ptr<connection_impl> conn)
        : pool_(pool), token_(token), conn_(conn)
    {
    }

    ~pooled_connection()
    {
        // If the connection is captured by a token then leave its release
        // to the token.
        if (!token_ && pool_) pool_free(pool_, conn_);
    }

    // Forcibly release the connection from the pool.
    void release()
    {
        if (token_)
        {
            token_->release(false);
            token_.reset();
        }

        // I couldn't come up with a normal case when a connection could have
        // a token but wasn't captured and released by it. But anyway let's
        // unconditionally release it here just to be sure.
        if (pool_) pool_free(pool_, conn_, false);
        pool_.reset();
    }

    boost::shared_ptr<connection_pool_impl> pool_;
    boost::shared_ptr<connection_pool_token> token_;
    boost::shared_ptr<connection_impl> conn_;
};

template <typename Handler>
inline void write_poll(const boost::shared_ptr<pooled_connection>& conn, Handler&& h)
{
    get_socket(conn->conn_)
        ->async_write_some(boost::asio::null_buffers(), std::forward<Handler>(h));
}

template <typename Handler>
inline void read_poll(const boost::shared_ptr<pooled_connection>& conn, Handler&& h)
{
    get_socket(conn->conn_)->async_read_some(boost::asio::null_buffers(), std::forward<Handler>(h));
}

inline boost::asio::io_service& get_io_service(const boost::shared_ptr<pooled_connection>& conn)
{
    return get_socket(conn->conn_)->get_io_service();
}

inline PGconn* get_connection(const boost::shared_ptr<pooled_connection>& conn)
{
    return conn->conn_->conn_.get();
}

inline const boost::shared_ptr<boost::asio::posix::stream_descriptor>& get_socket(
    const boost::shared_ptr<pooled_connection>& conn)
{
    return conn->conn_->sock_;
}

inline const boost::weak_ptr<yplatform::net::dns::resolver>& get_resolver(
    const boost::shared_ptr<pooled_connection>& conn)
{
    return conn->conn_->resolver_;
}

inline boost::shared_ptr<pooled_connection> pool_capture(
    boost::shared_ptr<connection_pool_impl>& impl,
    boost::shared_ptr<connection_pool_token>& token,
    boost::shared_ptr<connection_impl>& conn)
{
    if (token)
    {
        // Reserve the connection for a supplied token.
        boost::mutex::scoped_lock lock(impl->conn_mutex_);
        impl->captured_.insert(std::make_pair(reinterpret_cast<std::ptrdiff_t>(token.get()), conn));
    }
    return boost::make_shared<detail::pooled_connection>(impl, token, conn);
}

template <typename Handler>
class timed_handler
    : private boost::noncopyable
    , public boost::enable_shared_from_this<timed_handler<Handler>>
{
public:
    timed_handler(
        boost::shared_ptr<pooled_connection> conn,
        Handler handler,
        apq::time_traits::duration_type max_cancel_duration)
        : strand_(conn->pool_->queue_.get_io_service())
        , timer_(conn->pool_->queue_.get_io_service())
        , conn_(conn)
        , handler_(BOOST_ASIO_MOVE_CAST(Handler)(handler))
        , max_cancel_duration_(max_cancel_duration)
    {
    }

    template <typename Function>
    void perform(Function function, const apq::query& query, apq::time_traits::duration_type tm)
    {
        set_timer(adjust_for_cancel(tm));
        function(conn_, query, strand_.wrap(wrapper{ this->shared_from_this() }));
    }

    template <typename Function>
    void perform(
        Function function,
        const apq::query& query,
        apq::time_traits::duration_type tm,
        result_format rf)
    {
        set_timer(adjust_for_cancel(tm));
        function(conn_, query, strand_.wrap(wrapper{ this->shared_from_this() }), rf);
    }

private:
    apq::time_traits::duration_type adjust_for_cancel(apq::time_traits::duration_type t)
    {
        return t > max_cancel_duration_ ? t - max_cancel_duration_ : t;
    }

    void set_timer(apq::time_traits::duration_type tm)
    {
        if (tm != apq::time_traits::pos_infin_duration())
        {
            timer_.expires_from_now(tm);
            timer_.async_wait(strand_.wrap(boost::bind(
                &timed_handler<Handler>::handle_timer,
                this->shared_from_this(),
                boost::asio::placeholders::error)));
        }
    }

    struct wrapper
    {
        template <typename... Args>
        void operator()(Args&&... args) const
        {
            impl_->timer_.expires_at(apq::time_traits::pos_infin());
            impl_->handler_(std::forward<Args>(args)...);
        }

        template <typename Func>
        friend void asio_handler_invoke(Func&& f, wrapper* ctx)
        {
            using boost::asio::asio_handler_invoke;
            asio_handler_invoke(std::forward<Func>(f), &ctx->impl_->handler_);
        }

        boost::shared_ptr<timed_handler> impl_;
    };

    template <typename Func>
    friend void asio_handler_invoke(Func&& f, wrapper* ctx);

    void handle_timer(boost::system::error_code ec)
    {
        if (!ec && timer_.expires_at() != apq::time_traits::pos_infin())
        {
            conn_->release();
            close(conn_);
        }
    }

    boost::asio::io_service::strand strand_;
    apq::time_traits::timer_type timer_;
    boost::shared_ptr<pooled_connection> conn_;
    Handler handler_;
    apq::time_traits::duration_type max_cancel_duration_;
};

// Request function adaptor used to preserve completion handler type.
struct async_request_adaptor
{
    template <typename Handler>
    void operator()(
        boost::shared_ptr<pooled_connection> conn,
        const apq::query& q,
        Handler&& handler,
        result_format rf) const
    {
        async_op<request_cont>(
            conn,
            track_latency(std::forward<Handler>(handler), conn->conn_->average_db_latency_),
            q,
            rf);
    }
};

struct async_request_single_adaptor
{
    template <typename Handler>
    void operator()(
        boost::shared_ptr<pooled_connection> conn,
        const apq::query& q,
        Handler&& handler,
        result_format rf) const
    {
        // TODO: design statistics for single-row requests, tracking
        // whole request time does not make much sense
        async_op<single_tuple_request_cont>(conn, std::forward<Handler>(handler), q, rf);
    }
};

struct async_update_adaptor
{
    template <typename Handler>
    void operator()(
        boost::shared_ptr<pooled_connection> conn,
        const apq::query& q,
        Handler&& handler) const
    {
        async_op<update_cont>(
            conn,
            track_latency(std::forward<Handler>(handler), conn->conn_->average_db_latency_),
            q);
    }
};

struct async_execute_adaptor
{
    template <typename Handler>
    void operator()(
        boost::shared_ptr<pooled_connection> conn,
        const apq::query& q,
        Handler&& handler) const
    {
        async_op<execute_cont>(
            conn,
            track_latency(std::forward<Handler>(handler), conn->conn_->average_db_latency_),
            q);
    }
};

inline time_traits::duration_type effective_timeout(
    time_traits::duration_type timeout,
    const yplatform::task_context& ctx)
{
    if (ctx.deadline() < time_traits::time_type::max())
        return std::min(timeout, ctx.deadline() - time_traits::now());
    else
        return timeout;
}

template <typename Handler>
void pool_request(
    yplatform::const_task_context_ptr ctx,
    const apq::result& res,
    apq::connection_service::implementation_type conn,
    boost::shared_ptr<connection_pool_impl> impl,
    boost::shared_ptr<connection_pool_token> token,
    const apq::query& query,
    Handler handler,
    apq::time_traits::duration_type tm,
    result_format rf = result_format_text)
{
    assert(ctx != nullptr);
    if (res.code())
    {
        impl->queue_.get_io_service().post(boost::bind(handler, res, apq::row_iterator()));
        return;
    }
    // `conn` must be put (back) into its pool (`impl`) when no longer needed.
    auto pooled_conn = pool_capture(impl, token, conn);
    if (ctx->is_cancelled())
    {
        impl->queue_.get_io_service().post(
            boost::bind(handler, apq::result(error::task_cancelled), row_iterator{}));
        return;
    }
    boost::make_shared<timed_handler<Handler>>(pooled_conn, handler, impl->max_cancel_duration)
        ->perform(async_request_adaptor(), query, effective_timeout(tm, *ctx), rf);
}

template <typename Handler>
void pool_request_single(
    yplatform::const_task_context_ptr ctx,
    const apq::result& res,
    apq::connection_service::implementation_type conn,
    boost::shared_ptr<connection_pool_impl> impl,
    boost::shared_ptr<connection_pool_token> token,
    const apq::query& query,
    Handler handler,
    apq::time_traits::duration_type tm,
    result_format rf = result_format_text)
{
    assert(ctx != nullptr);
    if (res.code())
    {
        impl->queue_.get_io_service().post(boost::bind(handler, res, apq::cursor{}));
        return;
    }
    // `conn` must be put (back) into its pool (`impl`) when no longer needed.
    auto pooled_conn = pool_capture(impl, token, conn);
    if (ctx->is_cancelled())
    {
        impl->queue_.get_io_service().post(
            boost::bind(handler, apq::result(error::task_cancelled), cursor{}));
        return;
    }
    boost::make_shared<timed_handler<Handler>>(pooled_conn, handler, impl->max_cancel_duration)
        ->perform(async_request_single_adaptor(), query, effective_timeout(tm, *ctx), rf);
}

template <typename Handler>
void pool_update(
    yplatform::const_task_context_ptr ctx,
    const apq::result& res,
    apq::connection_service::implementation_type conn,
    boost::shared_ptr<connection_pool_impl> impl,
    boost::shared_ptr<connection_pool_token> token,
    const apq::query& query,
    Handler handler,
    apq::time_traits::duration_type tm)
{
    assert(ctx != nullptr);
    if (res.code())
    {
        impl->queue_.get_io_service().post(boost::bind(handler, res, 0));
        return;
    }
    // `conn` must be put (back) into its pool (`impl`) when no longer needed.
    auto pooled_conn = pool_capture(impl, token, conn);
    if (ctx->is_cancelled())
    {
        impl->queue_.get_io_service().post(
            boost::bind(handler, apq::result(error::task_cancelled), 0));
        return;
    }
    boost::make_shared<timed_handler<Handler>>(pooled_conn, handler, impl->max_cancel_duration)
        ->perform(async_update_adaptor(), query, effective_timeout(tm, *ctx));
}

template <typename Handler>
void pool_execute(
    yplatform::const_task_context_ptr ctx,
    const apq::result& res,
    apq::connection_service::implementation_type conn,
    boost::shared_ptr<connection_pool_impl> impl,
    boost::shared_ptr<connection_pool_token> token,
    const apq::query& query,
    Handler handler,
    apq::time_traits::duration_type tm)
{
    assert(ctx != nullptr);
    if (res.code())
    {
        impl->queue_.get_io_service().post(boost::bind(handler, res));
        return;
    }
    // `conn` must be put (back) into its pool (`impl`) when no longer needed.
    auto pooled_conn = pool_capture(impl, token, conn);
    if (ctx->is_cancelled())
    {
        impl->queue_.get_io_service().post(
            boost::bind(handler, apq::result(error::task_cancelled)));
        return;
    }
    boost::make_shared<timed_handler<Handler>>(pooled_conn, handler, impl->max_cancel_duration)
        ->perform(async_execute_adaptor(), query, effective_timeout(tm, *ctx));
}

} // namespace detail
} // namespace apq

#endif
