#pragma once

#include <string>
#include <boost/shared_ptr.hpp>
#include <boost/asio.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/make_shared.hpp>
#include <libpq-fe.h>
#include <pg_config.h>
#include <yplatform/net/dns/resolver.h>
#include <apq/detail/pq_init.hpp>
#include <apq/error.hpp>
#include <apq/query.hpp>
#include <apq/result.hpp>
#include <apq/row_iterator.hpp>
#include <apq/result_format.hpp>
#include <apq/time_traits.hpp>
#include <apq/detail/moving_average_duration.hpp>
#include <apq/detail/handler_stat.hpp>
#include <apq/detail/connection_info.hpp>
#include <apq/detail/connect_op.hpp>
#include <apq/detail/resolve_op.hpp>
#include <apq/detail/request_op.hpp>

#include <sys/socket.h> // for ::shutdown

namespace apq { namespace detail {

inline std::size_t count_commas(const char* s)
{
    static const size_t MAX_LEN = 1024;
    return std::count(s, s + strnlen(s, MAX_LEN), ',');
}

struct connection_impl
{
    using resolver_type = yplatform::net::dns::resolver;

    explicit connection_impl(boost::asio::io_service& ios)
        : strand_(boost::make_shared<boost::asio::io_service::strand>(ios))
        , sock_(boost::make_shared<boost::asio::posix::stream_descriptor>(ios))
        , close_time_(apq::time_traits::pos_infin())
    {
    }

    inline void start_connect(const connection_info& conninfo)
    {
        std::vector<const char*> keys, values;
        for (auto& [key, value] : conninfo)
        {
            keys.emplace_back(key.data());
            values.emplace_back(value.data());
        }
        keys.emplace_back(nullptr);
        values.emplace_back(nullptr);
        conn_.reset(PQconnectStartParams(keys.data(), values.data(), 0), PQfinish);
    }

    inline auto get_status()
    {
        return PQstatus(conn_.get());
    }

    inline auto get_error_message()
    {
        return PQerrorMessage(conn_.get());
    }

    inline auto connection_poll()
    {
        return PQconnectPoll(conn_.get());
    }

    inline bool is_connected()
    {
        return conn_.get() != nullptr;
    }

    std::tuple<boost::system::error_code, std::string> refresh_socket()
    {
        // No need to refresh anything for single-host conninfos.
        boost::system::error_code ec;
        if (!is_multihost_)
        {
            return std::make_tuple(ec, std::string{});
        }
        sock_->close(ec);
        if (ec)
        {
            return std::make_tuple(ec, "failed to close old socket");
        }
        return assign_socket();
    }

    std::tuple<boost::system::error_code, std::string> assign_socket()
    {
        int fd = PQsocket(conn_.get());
        if (fd == -1)
        {
            return std::make_tuple(
                error::network, "No server connection is currently open, no PQsocket");
        }

        int new_fd = dup(fd);
        if (new_fd == -1)
        {
            using namespace boost::system;
            return std::make_tuple(error_code{ errno, generic_category() }, "dup(fd)");
        }

        boost::system::error_code ec;
        sock_->assign(new_fd, ec);
        if (ec)
        {
            return std::make_tuple(ec, "assign(new_fd, ec)");
        }

        return std::make_tuple(ec, "");
    }

    void init_multihost()
    {
        static const char* KEYWORD_HOST = "host";
        static const char* KEYWORD_HOSTADDR = "hostaddr";

        auto conninfo_opts = std::unique_ptr<PQconninfoOption, void (*)(PQconninfoOption*)>(
            PQconninfo(conn_.get()), PQconninfoFree);
        if (!conninfo_opts)
        {
            is_multihost_ = false;
        }
        // https://www.postgresql.org/docs/10/static/libpq-connect.html
        // According to the docs, conninfo_opts is a pointer
        // to an array of PQconninfoOption structs, the last
        // has keyword set to null.
        const char* host = nullptr;
        const char* hostaddr = nullptr;
        for (auto opt = conninfo_opts.get(); opt->keyword; ++opt)
        {
            if (!strcmp(opt->keyword, KEYWORD_HOST))
            {
                host = opt->val;
            }
            if (!strcmp(opt->keyword, KEYWORD_HOSTADDR))
            {
                hostaddr = opt->val;
            }
        }
        if (!host && !hostaddr)
        {
            is_multihost_ = false;
        }
        // https://www.postgresql.org/docs/10/static/libpq-connect.html#LIBPQ-CONNECT-HOSTADDR
        // Hostaddr contains all the addresses we need; if it's missing - then use host.
        auto num_hosts = 1 + (hostaddr ? count_commas(hostaddr) : count_commas(host));
        is_multihost_ = num_hosts > 1;
    }

    boost::shared_ptr<PGconn> conn_;
    bool is_multihost_ = false;
    bool async_resolve_ = false;
    bool ipv6_only_ = false;
    boost::shared_ptr<boost::asio::io_service::strand> strand_;
    boost::shared_ptr<boost::asio::posix::stream_descriptor> sock_;
    boost::weak_ptr<resolver_type> resolver_;
    apq::time_traits::time_type close_time_;
    moving_average_duration_ptr average_db_latency_;
    apq::detail::pq_init<> init_;
};

template <typename Connection>
inline const boost::shared_ptr<boost::asio::posix::stream_descriptor>& get_socket(
    const boost::shared_ptr<Connection>& conn)
{
    return conn->sock_;
}

template <typename Connection>
inline const boost::weak_ptr<yplatform::net::dns::resolver>& get_resolver(
    const boost::shared_ptr<Connection>& conn)
{
    return conn->resolver_;
}

inline boost::asio::io_service& get_io_service(const boost::shared_ptr<connection_impl>& conn)
{
    return get_socket(conn)->get_io_service();
}

template <typename Handler>
inline void write_poll(const boost::shared_ptr<connection_impl>& conn, Handler&& h)
{
    get_socket(conn)->async_write_some(boost::asio::null_buffers(), std::forward<Handler>(h));
}

template <typename Handler>
inline void read_poll(const boost::shared_ptr<connection_impl>& conn, Handler&& h)
{
    get_socket(conn)->async_read_some(boost::asio::null_buffers(), std::forward<Handler>(h));
}

template <typename Handler>
void async_resolve(
    boost::shared_ptr<connection_impl> impl,
    const connection_info& conninfo,
    const Handler& handler)
{
    using resolver_type = connection_impl::resolver_type;
    auto resolver = boost::make_shared<resolver_type>(*impl->strand_);
    impl->resolver_ = resolver;
    auto op = std::make_shared<resolve_op<connection_impl, resolver_type>>(
        impl, resolver, conninfo, handler);
    yplatform::spawn(*impl->strand_, op);
}

template <typename Handler>
void async_connect(
    boost::shared_ptr<connection_impl>& impl,
    const std::string& conninfo_str,
    Handler handler)
{
    connection_info conninfo;
    try
    {
        conninfo = parse_connection_info(conninfo_str);
    }
    catch (const std::exception& e)
    {
        return handler(result(error::bad_conninfo, e.what()));
    }
    if (impl->async_resolve_)
    {
        async_resolve(impl, conninfo, [impl, handler](auto&& res, auto&& conninfo) mutable {
            if (res) return handler(res);
            connect_op<Handler, connection_impl>(impl, handler).perform(conninfo);
        });
    }
    else
    {
        connect_op<Handler, connection_impl>(impl, handler).perform(conninfo);
    }
}

template <template <class> class Cont, typename Connection, typename Handler, typename... Args>
void async_op(Connection&& impl, Handler&& handler, Args&&... args)
{
    using conn_t = typename std::decay<Connection>::type;
    using handler_t = typename std::decay<Handler>::type;
    using cont = Cont<handler_t>;
    using op = request_op<conn_t, cont>;

    boost::make_shared<op>(impl, cont(std::forward<Handler>(handler)))
        ->perform(std::forward<Args>(args)...);
}

template <typename Connection>
void cancel(Connection impl)
{
    boost::system::error_code ignored;
    get_socket(impl)->cancel(ignored);
    if (auto resolver = get_resolver(impl).lock()) resolver->cancel();
}

template <typename Connection>
void close(Connection impl)
{
    boost::system::error_code ignored;
    get_socket(impl)->close(ignored);
}

template <typename Connection>
void shutdown(Connection impl)
{
    // Boost's posix stream descriptor does not have a shutdown
    // method, therefore the native one is used.
    int sock = get_socket(impl)->native_handle();
    ::shutdown(sock, SHUT_RDWR);
}

} // namespace detail
} // namespace apq
