#ifndef APQ_MOVING_AVERAGE_HPP
#define APQ_MOVING_AVERAGE_HPP

#include <apq/time_traits.hpp>
#include <boost/circular_buffer.hpp>
#include <boost/thread.hpp>
#include <boost/type_traits/is_unsigned.hpp>
#include <boost/type_traits/is_same.hpp>
#include <boost/utility/enable_if.hpp>
#include <stdexcept>

namespace apq { namespace detail {

// XXX: sum will overflow if value * window_size is close to
// maximum ptrdiff_t (or size_t for unsigned values)
template <typename ValueType, typename LockGuard = boost::lock_guard<boost::mutex>>
class moving_average
{
public:
    typedef ValueType value_type;
    typedef LockGuard lock_guard_t;
    // int is chosen as window_size_type, because boost date_time
    // time_duration can be divided only by ints. Therefore,
    // if window size becomes larger than an int, division in
    // compute() will produce meaningless results
    typedef int window_size_t;

    moving_average(window_size_t window_size = 10000)
        : window_size_(window_size)
        , values_(static_cast<std::size_t>(window_size))
        , sum_(value_type())
    {
        if (window_size_ <= 0)
            throw std::invalid_argument("Invalid window size for moving_average!");
    }

    value_type compute() const
    {
        lock_guard_t lock(mutex_);
        return values_.empty() ?
            value_type() :
            static_cast<value_type>(sum_ / static_cast<window_size_t>(values_.size()));
    }

    void add_value(value_type val)
    {
        lock_guard_t lock(mutex_);
        if (static_cast<window_size_t>(values_.size()) == window_size_)
        {
            sum_ -= values_.front();
        }

        sum_ += val;
        values_.push_back(val);
    }

    void add_zero()
    {
        add_value(value_type());
    }

private:
    template <typename T, typename Enable = void>
    struct SumType
    {
        typedef std::ptrdiff_t type;
    };
    template <typename T>
    struct SumType<T, typename boost::enable_if<boost::is_unsigned<T>>::type>
    {
        typedef std::size_t type;
    };
    template <typename T>
    struct SumType<
        T,
        typename boost::enable_if<boost::is_same<T, time_traits::duration_type>>::type>
    {
        typedef T type;
    };

    typedef typename SumType<ValueType>::type sum_type;

    const window_size_t window_size_;

    boost::circular_buffer<value_type> values_;
    sum_type sum_;

    mutable typename lock_guard_t::mutex_type mutex_;
};

}}

#endif
