#pragma once

#include <apq/connection_service.hpp>

namespace apq::detail {

template <typename ConnectionPool, typename Handler>
struct pool_connect_op
    : public boost::enable_shared_from_this<pool_connect_op<ConnectionPool, Handler>>
{
    using this_type = pool_connect_op<ConnectionPool, Handler>;
    using connection_pool_type = ConnectionPool;
    using connection_type = typename connection_pool_type::connection_type;

    pool_connect_op(
        apq::connection_service& service,
        boost::shared_ptr<connection_pool_type>& impl,
        Handler handler)
        : timer_(impl->queue_.get_io_service()), service_(service), impl_(impl), handler_(handler)
    {
    }

    void perform()
    {
        service_.construct(conn_);
        conn_->average_db_latency_ = impl_->average_db_latency_;
        conn_->async_resolve_ = impl_->async_resolve_;
        conn_->ipv6_only_ = impl_->ipv6_only_;
        service_.async_connect(
            conn_,
            impl_->conninfo_,
            conn_->strand_->wrap(
                boost::bind(&this_type::handle_connect, this->shared_from_this(), _1)));

        set_timer();
    }

    void handle_connect(const apq::result& res)
    {
        timer_.expires_at(apq::time_traits::pos_infin());
        timer_.cancel();

        boost::mutex::scoped_lock lock(impl_->conn_mutex_);

        --impl_->temp_;

        if (res.code())
        {
            lock.unlock();
            handler_(res, conn_);
            return;
        }

        if (conn_->conn_.get() != NULL) impl_->busy_.insert(conn_);

        lock.unlock();

        handler_(boost::system::error_code(), conn_);
    }

    void handle_timer(const boost::system::error_code& ec)
    {
        if (!ec)
        {
            // Check whether the deadline has been moved to infinity before
            // this handler had a chance to run.
            if (timer_.expires_at() != apq::time_traits::pos_infin())
            {
                service_.cancel(conn_);
                // For multihost connections, give libpq another try. Thus
                // connect_timeout is silently multiplied by the number of hosts,
                // which is consistent with libpq behaviour.
                if (conn_->is_multihost_)
                {
                    set_timer();
                }
            }
        }
    }

    void set_timer()
    {
        timer_.expires_from_now(impl_->connect_timeout_);
        timer_.async_wait(conn_->strand_->wrap(boost::bind(
            &this_type::handle_timer, this->shared_from_this(), boost::asio::placeholders::error)));
    }

    apq::time_traits::timer_type timer_;
    apq::connection_service& service_;

    boost::shared_ptr<connection_pool_type> impl_;
    boost::shared_ptr<connection_type> conn_;
    Handler handler_;
};

}