#pragma once

#include <apq/row_iterator.hpp>
#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>
#include <libpq-fe.h>

namespace apq::detail {

typedef apq::row_iterator::result_ptr result_ptr;

template <typename Handler>
struct request_cont
{
    explicit request_cont(Handler handler) : handler_(std::move(handler))
    {
    }

    void process(result_ptr&& res)
    {
        res_ = std::move(res);
    }

    void done(apq::result res)
    {
        handler_(std::move(res), apq::row_iterator(std::move(res_)));
    }

    Handler handler_;
    result_ptr res_;
};

template <typename Handler>
struct single_tuple_request_cont
{
    explicit single_tuple_request_cont(Handler handler) : handler_(std::move(handler))
    {
    }

    void process(apq::cursor&& c)
    {
        handler_(apq::result{}, std::move(c));
    }

    void done(apq::result res)
    {
        handler_(std::move(res), apq::cursor{});
    }

    Handler handler_;
};

template <typename Handler>
struct update_cont
{
    explicit update_cont(Handler handler) : handler_(std::move(handler)), count_(0)
    {
    }

    void process(result_ptr&& res)
    {
        count_ += std::atoi(PQcmdTuples(res.get()));
    }

    void done(apq::result res)
    {
        const int count = res ? 0 : count_;
        handler_(std::move(res), count);
    }

    Handler handler_;
    int count_;
};

template <typename Handler>
struct execute_cont
{
    explicit execute_cont(Handler handler) : handler_(std::move(handler))
    {
    }

    void process(result_ptr&&) const
    {
    }

    void done(apq::result res)
    {
        handler_(std::move(res));
    }

    Handler handler_;
};

struct single_row_mode
{
};
struct default_row_mode
{
};

template <typename Cont>
inline default_row_mode row_mode(const Cont&)
{
    return {};
}

template <typename T>
inline single_row_mode row_mode(const single_tuple_request_cont<T>&)
{
    return {};
}

template <typename Connection, typename Cont>
struct request_op
    : boost::asio::coroutine
    , boost::enable_shared_from_this<request_op<Connection, Cont>>
    , cursor::continuation
{
    request_op(Connection& impl, Cont cont) : impl_(impl), cont_(std::move(cont))
    {
    }

    void perform(const std::string& query)
    {
        if (!PQsendQuery(get_connection(impl_), query.c_str()))
        {
            done(error::network, make_error_msg("PQsendQuery"));
            return;
        }

        handle();
    }

    void perform(const apq::query& q, result_format rf = result_format_text)
    {
        if (!q.values_.empty())
        {
            bind_parameters params = make_bind_parameters(q.values_);

            if (!PQsendQueryParams(
                    get_connection(impl_),
                    q.text_.c_str(),
                    static_cast<int>(params.values.size()),
                    params.types.data(),
                    params.values.data(),
                    params.lengths.data(),
                    params.formats.data(),
                    rf))
            {
                return done(error::network, make_error_msg("PQsendQueryParams"));
            }
        }
        else
        {
            const int res = (rf == result_format_text) ?
                PQsendQuery(get_connection(impl_), q.text_.c_str()) :
                PQsendQueryParams(
                    get_connection(impl_), q.text_.c_str(), 0, NULL, NULL, NULL, NULL, rf);
            if (!res)
            {
                return done(error::network, make_error_msg("PQsendQueryParams"));
            }
        }

        if (auto err = set_row_mode(row_mode(cont_)))
        {
            cancel(impl_);
            return done(std::move(err));
        }

        handle();
    }

    void handle(boost::system::error_code ec = boost::system::error_code{}, std::size_t = 0)
    {
        if (ec)
        {
            // Bad descriptor error can occur here if the connection was closed
            // by client during processing.
            if (ec == boost::asio::error::bad_descriptor)
                ec = make_error_code(boost::asio::error::operation_aborted);
            return done(ec);
        }

        try
        {
            result_ptr res;
            reenter(*this)
            {
                for (;;)
                {
                    yield wait_for_data();
                    if (auto err = consume())
                    {
                        return done(std::move(err));
                    }

                    while (!connection_busy())
                    {
                        res = get_result();
                        if (!res)
                        {
                            return done();
                        }

                        if (!succeeded(row_mode(cont_), *res))
                        {
                            return done_with_flush(
                                get_sql_error(*res), make_error_msg("request_op::get_result"));
                        }

                        if (single_row(row_mode(cont_)))
                        {
                            // Skip empty result in favor of final call with empty
                            // result in done() to avoid double final call of user
                            // handler in single row mode.
                            if (!empty(*res))
                            {
                                yield process(row_mode(cont_), std::move(res));
                            }
                        }
                        else
                        {
                            process(row_mode(cont_), std::move(res));
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            return done_with_flush(error::network, e.what());
        }
        catch (...)
        {
            return done_with_flush(
                error::network, std::string("unknown exception in ") + __PRETTY_FUNCTION__);
        }
    }

    apq::result set_row_mode(single_row_mode)
    {
        if (PQsetSingleRowMode(get_connection(impl_))) return {};
        return { error::network, make_error_msg("PQsetSingleRowMode") };
    }

    apq::result set_row_mode(default_row_mode) const
    {
        return {};
    }

    std::true_type single_row(single_row_mode) const
    {
        return {};
    }

    std::false_type single_row(default_row_mode) const
    {
        return {};
    }

    /**
     * This wrapper preserves handler context just like
     * boost::asio::detail::bind_handler does.
     * It is needed because in other case you will loose the strand
     * context which is used in timed_handler. See the link below:
     *
     *     https://stackoverflow.com/questions/32857101/when-to-use-asio-handler-invoke
     */
    template <typename Oper>
    struct wrapper
    {
        boost::shared_ptr<request_op> self;
        Oper oper;

        template <typename... Args>
        void operator()(Args&&... args)
        {
            oper(self, std::forward<Args>(args)...);
        }

        template <typename Func>
        friend void asio_handler_invoke(Func&& f, wrapper* ctx)
        {
            using boost::asio::asio_handler_invoke;
            asio_handler_invoke(std::forward<Func>(f), &ctx->self->cont_.handler_);
        }
    };

    template <typename Oper>
    wrapper<Oper> wrap(Oper oper)
    {
        return { this->shared_from_this(), std::move(oper) };
    }

    template <typename Oper>
    void post(Oper oper)
    {
        get_io_service(impl_).post(wrap(std::move(oper)));
    }

    void resume() override
    {
        post([](boost::shared_ptr<request_op> self) { self->handle(); });
    }

    void wait_for_data()
    {
        read_poll(
            impl_,
            wrap([](boost::shared_ptr<request_op> self,
                    boost::system::error_code e,
                    std::size_t n) { self->handle(e, n); }));
    }

    void process(single_row_mode, result_ptr&& res)
    {
        post([res = std::move(res)](boost::shared_ptr<request_op> self) mutable {
            self->cont_.process(cursor{ std::move(res), cursor::continuation_handle{ self } });
        });
    }

    template <typename... Args>
    void done(Args&&... args)
    {
        post([res = apq::result{ std::forward<Args>(args)... }](
                 boost::shared_ptr<request_op> self) mutable { self->cont_.done(std::move(res)); });
    }

    template <typename... Args>
    void done_with_flush(Args&&... args)
    {
        flush_result();
        done(std::forward<Args>(args)...);
    }

    void process(default_row_mode, result_ptr&& res)
    {
        cont_.process(std::move(res));
    }

    bool succeeded(default_row_mode, const PGresult& res) const
    {
        const auto s = PQresultStatus(&res);
        return s == PGRES_TUPLES_OK || s == PGRES_COMMAND_OK;
    }

    bool succeeded(single_row_mode, const PGresult& res) const
    {
        const auto s = PQresultStatus(&res);
        return s == PGRES_TUPLES_OK || s == PGRES_SINGLE_TUPLE;
    }

    result_ptr get_result() const
    {
        PGresult* res = PQgetResult(get_connection(impl_));
        return res ? result_ptr{ res, PQclear } : result_ptr{};
    }

    void flush_result() const
    {
        while (get_result())
            ;
    }

    char* get_sql_error(const PGresult& res) const
    {
        return PQresultErrorField(&res, PG_DIAG_SQLSTATE);
    }

    apq::result consume() const
    {
        if (!PQconsumeInput(get_connection(impl_)))
        {
            // Make up a generic "connection exception" SQLSTATE because
            // right now libpq doesn't provide a SQLSTATE for connections.
            return { error::network, make_error_msg("PQconsumeInput") };
        }
        return {};
    }

    bool connection_busy() const
    {
        return PQisBusy(get_connection(impl_));
    }

    std::string make_error_msg(const char* where) const
    {
        return ::apq::error::make_error_msg(where, PQerrorMessage(get_connection(impl_)));
    }

    Connection impl_;
    Cont cont_;
};

}

#include <boost/asio/unyield.hpp>