#pragma once

#include <apq/detail/connection_info.hpp>
#include <apq/result.hpp>
#include <yplatform/util/split.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <boost/algorithm/string/join.hpp>

namespace apq::detail {

template <typename Connection, typename Resolver>
struct resolve_op
{
    using yield_context = yplatform::yield_context<resolve_op>;
    using handler_type = std::function<void(result, const connection_info&)>;
    using iterator_a = typename Resolver::iterator_a;
    using iterator_aaaa = typename Resolver::iterator_aaaa;

    resolve_op(
        boost::shared_ptr<Connection> conn,
        boost::shared_ptr<Resolver> resolver,
        const connection_info& conninfo,
        const handler_type& handler)
        : conn_(conn), resolver_(resolver), conninfo_(conninfo), handler_(handler)
    {
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            if (already_resolved())
            {
                invoke_handler({}, conninfo_);
                yield break;
            }
            hosts = extract_hosts(conninfo_);
            reserve(resolved_hosts, hosts);
            reserve(resolved_host_addrs, hosts);
            for (host_it = hosts.begin(); host_it != hosts.end(); ++host_it)
            {
                yield resolver_->async_resolve_aaaa(*host_it, yield_ctx.capture(err, it_aaaa));
                if (err && err != boost::asio::error::host_not_found)
                {
                    invoke_handler(result(err), {});
                    yield break;
                }
                collect_resolved_addrs(*host_it, it_aaaa);
                if (conn_->ipv6_only_) continue;
                yield resolver_->async_resolve_a(*host_it, yield_ctx.capture(err, it_a));
                if (err && err != boost::asio::error::host_not_found)
                {
                    invoke_handler(result(err), {});
                    yield break;
                }
                collect_resolved_addrs(*host_it, it_a);
            }
            if (resolved_host_addrs.size())
            {
                conninfo_["host"] = boost::algorithm::join(resolved_hosts, ",");
                conninfo_["hostaddr"] = boost::algorithm::join(resolved_host_addrs, ",");
            }
            invoke_handler({}, conninfo_);
        }
    }

    void operator()(std::exception_ptr exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (const std::exception& e)
        {
            handler_(result(error::operation_exception, e.what()), {});
        }
    }

    bool already_resolved()
    {
        return conninfo_.count("hostaddr");
    }

    void invoke_handler(result res, const connection_info& conninfo)
    {
        yplatform::safe_call(handler_, res, conninfo);
    }

    auto extract_hosts(const connection_info& conninfo)
    {
        auto it = conninfo.find("host");
        if (it == conninfo.end()) return std::vector<std::string>();
        return yplatform::util::split(it->second, ",");
    }

    void reserve(std::vector<std::string>& resolved, const std::vector<std::string>& hosts)
    {
        if (conn_->ipv6_only_)
        {
            resolved.reserve(hosts.size());
        }
        else
        {
            resolved.reserve(2 * hosts.size());
        }
    }

    template <typename Iterator>
    void collect_resolved_addrs(const std::string& host, Iterator it)
    {
        for (; it != Iterator(); ++it)
        {
            resolved_hosts.emplace_back(host);
            resolved_host_addrs.emplace_back(*it);
        }
    }

    boost::shared_ptr<Connection> conn_;
    boost::shared_ptr<Resolver> resolver_;
    connection_info conninfo_;
    handler_type handler_;
    std::vector<std::string> hosts;
    std::vector<std::string>::iterator host_it;
    boost::system::error_code err;
    iterator_a it_a;
    iterator_aaaa it_aaaa;
    std::vector<std::string> resolved_hosts;
    std::vector<std::string> resolved_host_addrs;
};
}

#include <yplatform/unyield.h>