#ifndef APQ_QUERY_HPP
#define APQ_QUERY_HPP

#include <limits>
#include <apq/detail/bind.hpp>

namespace apq {

using detail::bind_value_type;
using detail::bind_pair_t;

class query
{
public:
    explicit query(const std::string& text) : text_(text)
    {
    }

    void bind_cref_string(const std::string& v)
    {
        push(bind_value_type::STRING, detail::make_cref_bind_value(v));
    }

    void bind_const_int64(int64_t v)
    {
        push(bind_value_type::INT64, detail::make_const_bind_value(v));
    }

    void bind_const_double(double v)
    {
        push(bind_value_type::DOUBLE, detail::make_const_bind_value(v));
    }

    void bind_const_string(std::string&& v)
    {
        push(bind_value_type::STRING, detail::make_const_bind_value(std::move(v)));
    }

    void bind_const_string(const std::string& v)
    {
        push(bind_value_type::STRING, detail::make_const_bind_value(v));
    }

    void bind_const_int64_vector(std::vector<int64_t>&& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::INT64_ARRAY, detail::make_const_bind_value(std::move(v)));
    }

    void bind_const_int64_vector(const std::vector<int64_t>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::INT64_ARRAY, detail::make_const_bind_value(v));
    }

    // Warning: This method is not safe, because PostgreSQL does not support
    // unsigned integers.
    void bind_const_int64_vector(std::vector<uint64_t>&& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::UINT64_ARRAY, detail::make_const_bind_value(std::move(v)));
    }

    void bind_const_int64_vector(const std::vector<uint64_t>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::UINT64_ARRAY, detail::make_const_bind_value(v));
    }

    void bind_const_string_vector(std::vector<std::string>&& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::STRING_ARRAY, detail::make_const_bind_value(std::move(v)));
    }

    void bind_const_string_vector(const std::vector<std::string>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::STRING_ARRAY, detail::make_const_bind_value(v));
    }

    void bind_cref_int64_vector(const std::vector<int64_t>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::INT64_ARRAY, detail::make_cref_bind_value(v));
    }

    // Warning: This method is not safe, because PostgreSQL does not support
    // unsigned integers.
    void bind_cref_int64_vector(const std::vector<uint64_t>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::UINT64_ARRAY, detail::make_cref_bind_value(v));
    }

    void bind_cref_string_vector(const std::vector<std::string>& v)
    {
        if (v.size() > std::numeric_limits<uint32_t>::max())
            throw std::runtime_error("Array is too long");
        push(bind_value_type::STRING_ARRAY, detail::make_cref_bind_value(v));
    }

    void bind_cptr_byte_array(const void* v, int sz)
    {
        push(bind_value_type::BYTE_ARRAY, detail::make_const_bytes_value(v, sz));
    }

    void bind_null()
    {
        push(bind_value_type::PGNULL, detail::make_const_bind_value(static_cast<int>(0)));
    }

    void bind_const_uuid(const boost::uuids::uuid uuid)
    {
        push(bind_value_type::UUID, detail::make_const_bind_value(uuid));
    }

    std::string text_;
    std::vector<bind_pair_t> values_;

private:
    void push(bind_value_type t, const boost::shared_ptr<detail::bind_value_base>& v)
    {
        if (values_.size() >= INT_MAX) throw std::runtime_error("Too many query parameters");
        values_.push_back(std::make_pair(t, v));
    }
};

} // namespace apq

#endif
