#ifndef APQ_ROW_ITERATOR_HPP
#define APQ_ROW_ITERATOR_HPP

#include <string>
#include <boost/iterator/iterator_facade.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <libpq-fe.h>
#include <apq/detail/bind_result.hpp>
#include <tuple>

namespace apq {

namespace detail {
inline int size(const PGresult& res)
{
    return PQntuples(&res);
}

inline bool empty(const PGresult& res)
{
    return 0 == size(res);
}
}

class row_adaptor
{
public:
    int size() const
    {
        return PQnfields(result_);
    }

    std::string at(int column) const
    {
        return std::string(PQgetvalue(result_, index_, check(column)));
    }

    std::string at(const char* name) const
    {
        return at(column_at(name));
    }

    std::string at(const std::string& name) const
    {
        return at(name.c_str());
    }

    template <typename T>
    void at(int column, T& t) const
    {
        detail::bind_result(t, result_, index_, check(column));
    }

    template <typename T>
    void at(const char* name, T& t) const
    {
        at(column_at(name), t);
    }

    template <typename T>
    void at(const std::string& name, T& t) const
    {
        at(name.c_str(), t);
    }

    bool is_null(int column) const
    {
        return PQgetisnull(result_, index_, check(column));
    }

    bool has_column(const char* name) const
    {
        return !out_of_range(column(name));
    }

    bool has_column(const std::string& name) const
    {
        return has_column(name.c_str());
    }

    row_adaptor(const PGresult* result, int index) : result_(result), index_(index)
    {
    }

private:
    bool out_of_range(int column) const
    {
        return column < 0 || column >= size();
    }

    int column(const char* name) const
    {
        return PQfnumber(result_, name);
    }

    int column_at(const char* name) const
    {
        const int i = column(name);
        if (out_of_range(i))
        {
            throw std::out_of_range(std::string("column name \"") + name + "\" not found");
        }
        return i;
    }

    int check(int column) const
    {
        if (out_of_range(column))
        {
            throw std::out_of_range(
                std::string("column number ") + std::to_string(column) + " is out of range");
        }
        return column;
    }

    const PGresult* result_;
    const int index_;
};

class row_iterator
    : public boost::
          iterator_facade<row_iterator, row_adaptor, boost::forward_traversal_tag, row_adaptor>
{
public:
    typedef boost::shared_ptr<PGresult> result_ptr;

    row_iterator() = default;

    row_iterator(result_ptr result)
        : result_(result && !detail::empty(*result) ? std::move(result) : nullptr)
    {
    }

    operator bool() const
    {
        return result_.get();
    }

private:
    void increment()
    {
        if (++index_ == detail::size(*result_))
        {
            // Reset state to match a default constructed end iterator.
            result_.reset();
            index_ = 0;
        }
    }

    bool equal(const row_iterator& other) const
    {
        if (!(*this) && !other) return true;
        if (result_ != other.result_) return false;
        return index_ == other.index_;
    }

    reference dereference() const
    {
        return row_adaptor(result_.get(), index_);
    }

    result_ptr result_;
    int index_ = 0;

    friend class boost::iterator_core_access;
};

class cursor
{
public:
    struct continuation
    {
        virtual void resume() = 0;
        virtual ~continuation() = default;
    };

    class continuation_handle
    {
        using continuation_ptr = boost::shared_ptr<continuation>;
        continuation_ptr cont_;

    public:
        continuation_handle() = default;
        continuation_handle(continuation_ptr cont) : cont_(cont)
        {
        }
        continuation_handle(const continuation_handle&) = delete;
        continuation_handle(continuation_handle&&) = default;
        continuation_handle& operator=(const continuation_handle&) = delete;
        continuation_handle& operator=(continuation_handle&&) = default;

        void resume()
        {
            if (!*this) throw std::logic_error("call resume() for false handle");
            cont_->resume();
            cont_.reset();
        }

        operator bool() const noexcept
        {
            return bool(cont_);
        }
    };

    using context = std::tuple<row_iterator, continuation_handle>;

    cursor() = default;
    cursor(row_iterator r, continuation_handle c)
    {
        if (!r) throw std::invalid_argument("invalid row_iterator");
        if (!c) throw std::invalid_argument("invalid continuation_handle");
        ctx_ = boost::make_shared<context>(std::move(r), std::move(c));
    }

    boost::shared_ptr<context> get()
    {
        return std::move(ctx_);
    }

private:
    boost::shared_ptr<context> ctx_;
};

inline cursor::continuation_handle& continuation(cursor::context& ctx)
{
    return std::get<1>(ctx);
}

inline row_iterator& row(cursor::context& ctx)
{
    return std::get<0>(ctx);
}

} // namespace apq

#endif
