#ifndef APQ_SERIALIZE_COMPOSITE_HPP
#define APQ_SERIALIZE_COMPOSITE_HPP

#include <string>
#include <sstream>
#include <boost/type_traits.hpp>
#include <boost/mpl/or.hpp>
#include <boost/mpl/bool.hpp>
#include <boost/fusion/algorithm/iteration/for_each.hpp>
#include <boost/range/iterator.hpp>
#include <boost/range/as_literal.hpp>

namespace apq {

namespace detail {

template <typename T>
struct is_string
    : boost::mpl::or_<
          boost::is_same<char*, typename boost::decay<T>::type>,
          boost::is_same<const char*, typename boost::decay<T>::type>>
{
};

template <typename Char, typename Traits, typename Allocator>
struct is_string<std::basic_string<Char, Traits, Allocator>> : boost::mpl::true_
{
};

class serializer
{
public:
    explicit serializer(std::ostream& os) : os_(os), first_(true)
    {
    }

    template <typename T>
    typename boost::enable_if<is_string<T>>::type operator()(const T& x) const
    {
        write_comma();
        os_ << "\"";
        write_escaped(boost::as_literal(x));
        os_ << "\"";
    }

    template <typename T>
    typename boost::disable_if<is_string<T>>::type operator()(const T& x) const
    {
        write_comma();
        os_ << x;
    }

private:
    void write_comma() const
    {
        if (!first_) os_ << ",";
        else
            first_ = false;
    }

    template <typename Range>
    void write_escaped(const Range& r) const
    {
        for (typename boost::range_iterator<const Range>::type it = r.begin(); it != r.end(); ++it)
        {
            if (*it == '"' || *it == '\\') os_ << "\\";
            os_ << *it;
        }
    }

    std::ostream& os_;
    mutable bool first_;
};

} // namespace detail

// Dummy class to represent null value.
struct serializable_null
{
};

inline std::ostream& operator<<(std::ostream& os, const serializable_null&)
{
    return os;
}

template <typename T>
inline std::string serialize_composite(const T& t)
{
    std::ostringstream ss;
    ss << "(";
    boost::fusion::for_each(t, detail::serializer(ss));
    ss << ")";
    return ss.str();
}

} // namespace apq

#endif
