#ifndef APQ_TIME_TRAITS_HPP
#define APQ_TIME_TRAITS_HPP

#include <boost/asio/steady_timer.hpp>

namespace apq {

struct time_traits
{
    // The duration time of the clock.
    typedef boost::asio::steady_timer::duration duration_type;

    // The time point time of the clock.
    typedef boost::asio::steady_timer::time_point time_type;

    // The corresponding timer type.
    typedef boost::asio::steady_timer timer_type;

    // Milliseconds type
    typedef std::chrono::milliseconds milliseconds;

    // Get the current time.
    static time_type now()
    {
        return boost::asio::steady_timer::clock_type::now();
    }

    // Get a time value that never fires.
    static time_type pos_infin()
    {
        return time_type::max();
    }

    // Get a largest possible duration value.
    static duration_type pos_infin_duration()
    {
        return duration_type::max();
    }

    // Add a duration to a time while working around overflows.
    static time_type add(const time_type& t, const duration_type& d)
    {
        const time_type epoch;
        if (t >= epoch)
        {
            if (time_type::max() - t < d) return time_type::max();
        }
        else
        {
            if (-(t - time_type::min()) > d) return time_type::min();
        }
        return t + d;
    }

    // Convert duration to microseconds
    static duration_type::rep to_usec(const duration_type& d)
    {
        return std::chrono::duration_cast<std::chrono::microseconds>(d).count();
    }
};

} // namespace apq

#endif
