import os

import yaml

from library.python import resource
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import YplatformComponent
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.tvmapi import TvmApi
from mail.pg.queuedb.devpack.components.queuedb import QueueDb
from mail.webmail_config.lib.make_config import make_config
from .nwbbmock import NwBbMock


class BarbetApi(object):
    def __init__(self, yhttp, **default_args):
        self.yhttp = yhttp
        self.default_args = default_args

    def ping(self):
        return self.yhttp.ping()

    def create(self, uid, **common_args):
        return self.yhttp.post('/backup/create', data={'uid': uid}, **common_args)

    def delete(self, uid, **common_args):
        return self.yhttp.post('/backup/delete', data={'uid': uid}, **common_args)

    def restore(self, uid, method, **common_args):
        return self.yhttp.post('/backup/restore', data={'uid': uid, 'method': method}, **common_args)

    def settings(self, uid, **common_args):
        return self.yhttp.get('/backup/settings', params={'uid': uid}, **common_args)

    def update_settings(self, uid, fids=None, tabs=None, **common_args):
        data = {'uid': uid}
        if fids:
            data['fids'] = fids
        if tabs:
            data['tabs'] = tabs
        return self.yhttp.post('/backup/update_settings', data=data, **common_args)

    def status(self, uid, **common_args):
        return self.yhttp.get('/backup/status', params={'uid': uid}, **common_args)

    def archive_restore(self, uid, **common_args):
        return self.yhttp.post('/archive/restore', data={'uid': uid}, **common_args)


class BarbetBase(YplatformComponent):
    def __init__(self, env, components):
        super(BarbetBase, self).__init__(env, components, binary_name='barbet', custom_path='barbet')

    def _generate_config(self, env):
        base = resource.find('barbet/base.yml')
        service_yaml = yaml.safe_load(resource.find('barbet/service.yaml'))
        common_yaml = yaml.safe_load(resource.find('webmail_config/common.yaml'))
        queuedb_yaml = yaml.safe_load(resource.find('ymod_queuedb/service.yaml'))
        cfg = make_config(env, base, service_yaml, common_yaml, queuedb_yaml, silent=True)
        config = self.yhttp.format_config(cfg)
        helpers.write2file(config, os.path.join(self.etc_path, 'barbet.yml'))

    def replace_config(self):
        pass

    def init_root(self):
        self.yhttp.init_root()

        helpers.write2file(resource.find('barbet/tvm_secret'), os.path.join(self.etc_path, 'tvm_secret'))

        etc_macs_pg_path = os.path.join(self.get_root(), 'etc', 'macs_pg')
        helpers.mkdir_recursive(etc_macs_pg_path)
        helpers.write2file(resource.find('macs_pg/query.conf'), os.path.join(etc_macs_pg_path, 'query.conf'))

        etc_queuedb_path = os.path.join(self.get_root(), 'etc', 'queuedb')
        helpers.mkdir_recursive(etc_queuedb_path)
        helpers.write2file(resource.find('ymod_queuedb/queuedb.conf'), os.path.join(etc_queuedb_path, 'queuedb.conf'))

        self.replace_config()

        self.yhttp.init_pgpass(self.get_root())

        self._generate_config('development')

    def start(self):
        self.yhttp.put_pgpassfile_in_env(self.get_root())
        self.yhttp.start('pong')


class BarbetDevpack(BarbetBase):
    NAME = 'barbet_devpack'
    DEPS = [Sharpei, QueueDb, NwBbMock, TvmApi]

    def __init__(self, env, components):
        super(BarbetDevpack, self).__init__(env, components)
        self.__pyremock_port = self.config[self.NAME]['pyremock_port']
        self.__archive_restore_max_tries = 12
        self.__s3_bucket = 'mail-archive-test'

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(BarbetDevpack, cls).gen_config(port_generator, config=config)
        base.update({'pyremock_port': next(port_generator)})
        return base

    def replace_config(self):
        pyremock_url = 'http://localhost:{port}'.format(port=self.pyremock_port())
        replace = {
            'tvmapi_port': self.components[TvmApi].port,
            'sharpei_port': self.components[Sharpei].webserver_port(),
            'connection_string': f'host=localhost port={self.components[QueueDb].port()} user=barbet dbname=queuedb',
            'blackbox_url': f'http://localhost:{self.components[NwBbMock].port}',
            'nwsmtp_url': f'http://localhost:{self.components[NwBbMock].port}',
            's3_url': pyremock_url,
            'hound_url': pyremock_url,

            's3_bucket': self.s3_bucket(),
            'archive_restore_max_tries': self.archive_restore_max_tries()
        }
        devpack = self.yhttp.format_config(resource.find('barbet/local_mdb_queuedb.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.etc_path, 'barbet-devpack.yml'))

    def api(self, **default_args):
        return BarbetApi(self.yhttp, **default_args)

    def pyremock_port(self):
        return self.__pyremock_port

    def s3_bucket(self):
        return self.__s3_bucket

    def archive_restore_max_tries(self):
        return self.__archive_restore_max_tries


class BarbetLocalQueuedb(BarbetBase):
    NAME = 'barbet_local_queuedb'
    DEPS = [QueueDb]

    def __init__(self, env, components):
        super(BarbetLocalQueuedb, self).__init__(env, components)

    def replace_config(self):
        replace = {
            'connection_string': f'host=localhost port={self.components[QueueDb].port()} user=barbet dbname=queuedb',
        }
        devpack = self.yhttp.format_config(resource.find('barbet/local_queuedb.yml'), **replace)
        helpers.write2file(devpack, os.path.join(self.etc_path, 'barbet-devpack.yml'))
