import os

import requests
from werkzeug import exceptions as http_exceptions
from werkzeug.wrappers import Response, Request
from werkzeug.wsgi import responder

import json
from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import BaseComponent, WithPort
from mail.devpack.lib.components.sharpei import Sharpei
from mail.devpack.lib.components.mdb import Mdb
from mail.devpack.lib.errors import DevpackError


class MockApp(object):
    def __init__(self, sharpei, mdb):
        self.sharpei = sharpei
        self.mdb = mdb

    def ping(self, request):
        return Response('ok', mimetype='text/plain')

    def restore(self, request):
        uid = int(request.args['uid'])
        req = json.loads(request.get_data(as_text=True))
        st_id = req['mail_info']['stid']
        if 'absent_stid' in st_id:
            return Response(json.dumps(dict(code='StorageMailNotFound')), status=406, mimetype='application/json')
        if 'bad_stid' in st_id:
            return Response(json.dumps(dict(code='NslsPermanentError')), status=406, mimetype='application/json')

        shard_id = self.sharpei.api().conninfo(uid, 'write_only').json()['id']
        mid = self.mdb.shard_by_id(shard_id).query(
            f'''
                SELECT res.mid
                FROM code.store_message_with_tab(
                            i_uid =>{uid}::bigint,
                            i_coords => (
                                {req['mail_info']['fid']}::bigint, null::bigint, {'seen_label' in req['mail_info']['symbol']}, false,
                                '{req['mail_info']['stid']}'::text,
                                to_timestamp({req['mail_info']['received_date']}::double precision),
                                1024, array[]::mail.message_attributes[], '',
                                '{req['mail_info']['tab']}'::mail.tab_types
                            )::code.store_coordinates_with_tab,
                            i_headers => (
                                'Restored subject'::text, 'Restored firstline'::text,
                                to_timestamp({req['mail_info']['received_date']}::double precision),
                                '<restored-message-id>'::text, ''
                            )::code.store_headers,
                            i_recipients => array[]::code.store_recipient[],
                            i_attaches => array[]::code.store_attach[],
                            i_lids => array[]::integer[],
                            i_threads => (
                                'hash'::mail.threads_merge_rules,
                                array[]::numeric[], null, 0::numeric,
                                'subject'::mail.thread_hash_namespaces,
                                0::bigint, ''
                            )::code.store_threading
                    ) AS res
            '''
        )[0][0]
        return Response(json.dumps(dict(mid=str(mid))), mimetype='application/json')

    def blackbox(self, request):
        ret = {
            'users': [{
                'address-list': [{
                    'address': 'devnull@ya.ru',
                    'default': True,
                }]
            }]
        }
        return Response(json.dumps(ret), mimetype='application/json')

    @responder
    def __call__(self, environ, start_response):
        request = Request(environ)
        handlers = {
            u'/ping': self.ping,
            u'/mail/restore': self.restore,
            u'/blackbox': self.blackbox,
        }

        for path, h in handlers.items():
            if request.path.startswith(path):
                return h(request)
        return http_exceptions.NotFound()


class NwBbMock(BaseComponent, WithPort):
    NAME = "nwbbmock"
    DEPS = [Sharpei]

    def __init__(self, env, components):
        super(NwBbMock, self).__init__(env, components)
        self.app = MockApp(sharpei=self.components[Sharpei], mdb=self.components[Mdb])

    def start(self):
        self.logger.info('starting nwbbmock on port %d', self.port)
        child_pid = os.fork()
        if child_pid:
            try:
                helpers.wait_ping(self.logger, self.ping)
            except:
                raise DevpackError("nwbbmock does not response to /ping after start")
            self.state["pid"] = child_pid
            self.logger.info('nwbbmock started, pid: %d', child_pid)
        else:
            helpers.start_werkzeug(self)

    def stop(self):
        pid = self.state.get("pid")
        if pid:
            self.logger.info('stopping nwbbmock on port %d with pid %d', self.port, pid)
            helpers.kill_proc(pid)
            del self.state["pid"]
            self.logger.info('nwbbmock stopped')

    def info(self):
        return {
            "state": self.state,
            "root": self.root,
            "port": self.port,
        }

    def _request(self, query, timeout=10):
        url = "http://localhost:{port}{query}".format(port=self.port, query=query)
        self.logger.info('http call: %s', url)
        return requests.get(url, timeout=timeout)

    def ping(self):
        return self._request('/ping')
