#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <tvm_guard/module.h>

#include <mail/barbet/service/include/log.h>
#include <mail/barbet/service/include/handlers/status.h>
#include <mail/barbet/service/include/handlers/settings.h>
#include <mail/barbet/service/include/handlers/backup.h>
#include <mail/barbet/service/include/handlers/restore.h>
#include <mail/barbet/service/include/handlers/archive.h>

#include <mail/barbet/http_api/include/response.h>
#include <mail/barbet/http_api/include/parse.h>

#include <mail/http_getter/client/include/endpoint_reflection.h>

#include <mail/webmail/http_api_helpers/include/handler_helpers.h>
#include <mail/webmail/http_api_helpers/include/ping.h>

#include <mail/ymod_maildb/include/module.h>

#include <mail/ymod_queuedb/include/queue.h>

#include <yamail/data/deserialization/ptree_reader.h>

#include <pa/async.h>


namespace barbet {

struct HttpApi: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        const auto server = findDependency<ymod_webserver::server>(cfg, "dependencies.server");
        const auto guarded = findDependency<tvm_guard::Module>(cfg, "dependencies.guard");
        const auto maildb = findDependency<ymod_maildb::Module>(cfg, "dependencies.maildb");
        const auto queuedb = findDependency<ymod_queuedb::Queue>(cfg, "dependencies.queuedb");
        const auto getter = findDependency<http_getter::ClientModule>(cfg, "dependencies.http_getter");
        const auto reactor = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");

        const auto houndArchiveEndpoint = yamail::data::deserialization::fromPtree<http_getter::Endpoint>(
            cfg.get_child("hound_archive")
        );

        const bool guardedBySetting = cfg.get<bool>("guarded_by_setting");
        const auto attributes = boost::coroutines::attributes(cfg.get<unsigned>("coroutine_stack_size"));
        const auto maxMessages = cfg.get<unsigned>("create_backup.max_messages");

        unsigned secondsBetweenBackups = yplatform::time_traits::duration_cast<std::chrono::seconds>(
            cfg.get<yplatform::time_traits::duration>("create_backup.between_backups_dur")
        ).count();

        pa::async_profiler::init(500000, 16, cfg.get<std::string>("profiler_log"));
        ymod_queuedb::Timeout timeout(std::chrono::seconds(5));

        http_api::BindInfo<Response> info {
            .guarded=guarded,
            .server=server,
            .attributes=attributes,
            .reactor=reactor,
        };

        http_api::bindPingHandler(info);

        bindPOST<archive::RestoreResult>(info, "/archive/restore",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return archive::restore(
                    archive::parseRestoreParams(s, *maildb, *getter, houndArchiveEndpoint, timeout, *queuedb),
                    yield);
        });

        bindPOST<archive::DiscardResult>(info, "/archive/discard",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return archive::discard(archive::parseDiscardParams(s, *maildb, *getter, houndArchiveEndpoint), yield);
        });

        bindGET<StatusResult>(info, "/backup/status",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return status(parseStatusParams(s, *maildb), yield);
        });

        bindGET<SettingsResult>(info, "/backup/settings",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return settings(parseSettingsParams(s, *maildb), yield);
        });

        bindPOST<SettingsResult>(info, "/backup/update_settings",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return updateSettings(parseUpdateSettingsParams(s, *maildb), yield);
        });

        bindPOST<CreateBackupResult>(info, "/backup/create",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                CreateBackupParams params = parseCreateBackupParams(
                    s, maxMessages, timeout, secondsBetweenBackups,
                    guardedBySetting, *queuedb, *maildb
                );

                return createBackup(std::move(params), yield);
        });

        bindPOST<RestoreResult>(info, "/backup/restore",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                RestoreParams params = parseRestoreParams(
                    s, timeout, guardedBySetting, *queuedb, *maildb
                );

                return restore(std::move(params), yield);
        });

        bindPOST<DeactivateBackupResult>(info, "/backup/delete",
            [=](ymod_webserver::response_ptr s, boost::asio::yield_context yield) {
                return deactivateBackup(parseDeactivateBackupParams(s, *maildb), yield);
        });

        LOGDOG_(getModuleLogger(), notice, log::message="barbet::HttpApi loaded");
    }
};

}

DEFINE_SERVICE_OBJECT(barbet::HttpApi)
