#include <mail/barbet/http_api/include/parse.h>
#include <mail/barbet/http_api/include/context.h>
#include <mail/barbet/service/include/log.h>
#include <mail/barbet/service/include/error.h>
#include <macs_pg/service/factory.h>
#include <macs/backup_reflection.h>


namespace barbet {
namespace {
ymod_queuedb::Uid parseUid(const std::string& uid) {
    try {
        return ymod_queuedb::Uid(std::stoll(uid));
    } catch (const std::exception&) {
        throw mail_errors::system_error(make_error(http_api::Error::invalidArgument, "uid"));
    }
}

macs::ServicePtr service(const std::string& uid, const std::string& reqId,
                         const ymod_webserver::response& s,
                         const ymod_maildb::Module& maildb) {
    return maildb.service(
        ymod_maildb::parseServiceParams(uid, "barbet", s),
        ymod_maildb::parseUserJournalParams(s),
        std::make_shared<MacsPgLog>(getContextLogger(uid, reqId)),
        macs::pg::readNoLagReplicaThenMasterThenReplica
    );
}

macs::ServicePtr master(const std::string& uid, const std::string& reqId,
                        const ymod_webserver::response& s,
                        const ymod_maildb::Module& maildb) {
    return maildb.service(
        ymod_maildb::parseServiceParams(uid, "barbet", s),
        ymod_maildb::parseUserJournalParams(s),
        std::make_shared<MacsPgLog>(getContextLogger(uid, reqId)),
        macs::pg::masterOnly
    );
}
}

StatusParams parseStatusParams(ymod_webserver::response_ptr s, const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");

    return StatusParams {
        .service=service(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId)
    };
}

SettingsParams parseSettingsParams(ymod_webserver::response_ptr s, const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");

    return SettingsParams {
        .service=service(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId)
    };
}

DeactivateBackupParams parseDeactivateBackupParams(ymod_webserver::response_ptr s,
                                                   const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");

    return DeactivateBackupParams {
        .service=master(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId)
    };
}

UpdateSettingsParams parseUpdateSettingsParams(ymod_webserver::response_ptr s, const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");

    return UpdateSettingsParams {
        .service=master(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId),
        .fids=ctx.optionalArgs("fids").value_or(std::vector<std::string>()),
        .tabs=ctx.optionalArgs("tabs").value_or(std::vector<std::string>())
    };
}

RestoreParams parseRestoreParams(ymod_webserver::response_ptr s,
                                 ymod_queuedb::Timeout timeout,
                                 bool guardedBySetting,
                                 const ymod_queuedb::Queue& queuedb,
                                 const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");
    const auto sr = ymod_maildb::parseServiceParams(uid, "barbet", *s);
    const auto uj = ymod_maildb::parseUserJournalParams(*s);
    const auto logger = getContextLogger(uid, reqId);

    const std::string method = ctx.arg("method");
    RestoreParams::Kind kind;
    if (!yamail::data::reflection::from_string<RestoreParams::Kind>(method, kind)) {
        throw mail_errors::system_error(make_error(ServiceError::unknownRestoreType, method));
    }

    return RestoreParams {
        .service=maildb.service(
            sr, uj, std::make_shared<MacsPgLog>(logger),
            macs::pg::masterOnly),
        .logger=logger,
        .uid=parseUid(uid),
        .timeout=timeout,
        .queuedb=queuedb,
        .macsParams={ .sr=sr, .uj=uj },
        .kind=kind,
        .guardedBySetting=guardedBySetting
    };
}

CreateBackupParams parseCreateBackupParams(ymod_webserver::response_ptr s,
                                           unsigned maxMessages,
                                           ymod_queuedb::Timeout timeout,
                                           unsigned secondsBetweenBackups,
                                           bool guardedBySetting,
                                           const ymod_queuedb::Queue& queuedb,
                                           const ymod_maildb::Module& maildb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");
    const auto sr = ymod_maildb::parseServiceParams(uid, "barbet", *s);
    const auto uj = ymod_maildb::parseUserJournalParams(*s);
    const auto logger = getContextLogger(uid, reqId);

    return CreateBackupParams {
        .service=maildb.service(
            sr, uj, std::make_shared<MacsPgLog>(logger),
            macs::pg::masterOnly),
        .logger=logger,
        .uid=parseUid(uid),
        .timeout=timeout,
        .queuedb=queuedb,
        .macsParams={ .sr=sr, .uj=uj },
        .maxMessages=maxMessages,
        .secondsBetweenBackups=secondsBetweenBackups,
        .guardedBySetting=guardedBySetting
    };
}

namespace archive {

DiscardParams parseDiscardParams(ymod_webserver::response_ptr s,
                                 const ymod_maildb::Module& maildb,
                                 const http_getter::ClientModule& getter,
                                 const http_getter::Endpoint& houndEndpoint) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");

    return DiscardParams {
        .service=master(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId),
        .uid=uid,
        .client=getter.create(
            reqId, http_getter::withLog(getter.httpLogger(uid, reqId))
        ),
        .hound=houndEndpoint
    };
}

RestoreParams parseRestoreParams(ymod_webserver::response_ptr s,
                                        const ymod_maildb::Module& maildb,
                                        const http_getter::ClientModule& getter,
                                        const http_getter::Endpoint& houndEndpoint,
                                        ymod_queuedb::Timeout timeout,
                                        const ymod_queuedb::Queue& queuedb) {
    Context ctx{s->request()};
    const auto uid = ctx.arg("uid");
    const auto reqId = ctx.optionalHeader("x-request-id").value_or("");
    const auto sr = ymod_maildb::parseServiceParams(uid, "barbet", *s);
    const auto uj = ymod_maildb::parseUserJournalParams(*s);

    return RestoreParams {
        .service=master(uid, reqId, *s, maildb),
        .logger=getContextLogger(uid, reqId),
        .uid=uid,
        .client=getter.create(
            reqId, http_getter::withLog(getter.httpLogger(uid, reqId))
        ),
        .hound=houndEndpoint,
        .timeout=timeout,
        .queuedb=queuedb,
        .macsParams={ .sr=sr, .uj=uj }
    };
}

}

}
