#include <mail/barbet/service/include/delivery.h>
#include <mail/barbet/service/include/params.h>
#include <mail/barbet/service/include/helpers.h>
#include <yamail/data/deserialization/json_reader.h>
#include <yamail/data/serialization/json_writer.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>

namespace barbet {

using namespace std::string_literals;

namespace {

class FidService : public FidProvider {
private:
    macs::ServicePtr service;
    macs::Fid defaultFid;
    std::set<macs::Fid> createdTemporaryFids;
    std::string folderName;
    macs::BackupFidsMapping mapping;
public:
    FidService(macs::ServicePtr service, macs::Fid defaultFid, std::string folderName, macs::BackupFidsMapping mapping)
        : service(std::move(service))
        , defaultFid(std::move(defaultFid))
        , folderName(std::move(folderName))
        , mapping(std::move(mapping))
        { }

    macs::Fid temporaryFid(YieldCtx yield) override {
        macs::Fid restored = macs::getOrCreateFolderBySymbolWithRandomizedName(service->folders(),
            "Restored", macs::Folder::noParent, macs::Folder::Symbol::restored, false, io_result::make_yield_context(yield)
        ).fid();

        macs::Fid temporaryFid = service->folders().getOrCreateFolder(
            folderName,
            restored,
            macs::Folder::Symbol::none,
            io_result::make_yield_context(yield)
        ).fid();

        createdTemporaryFids.insert(temporaryFid);

        return temporaryFid;
    }

    const macs::Fid& getDefaultFid() const override {
        return defaultFid;
    }

    macs::Fid resolveFid(const macs::Fid& fid) const override {
        return helpers::resolveFid(fid, defaultFid, mapping);
    }

    void clearTemporaryFids(YieldCtx yield) override {
        if(createdTemporaryFids.empty()) {
            return;
        }

        const macs::FoldersRepository& repo = service->folders();
        const macs::FolderSet foldersWithTemporary = repo.getAllFoldersWithHidden(io_result::make_yield_context(yield));
        for (const auto& temporaryFid: createdTemporaryFids) {
            repo.erase(foldersWithTemporary.at(temporaryFid));
        }

        createdTemporaryFids.clear();
    }

    virtual ~FidService() = default;
};


class EnvelopeService : public EnvelopeProvider {
private:
    macs::ServicePtr service;
public:
    EnvelopeService(macs::ServicePtr service) : service(std::move(service)) { }

    macs::Stid stidByMid(const macs::Mid& mid, YieldCtx yield) const override {
        service->labels().resetLabelsCache();
        return service->envelopes().getById(mid, io_result::make_yield_context(yield)).stid();
    }

    void moveMessage(const macs::Mid& mid, const std::optional<macs::Tab::Type>& tab, const macs::Fid& fid, YieldCtx yield) const override {
        service->envelopes().moveMessages(
            fid, tab, {mid}, io_result::make_yield_context(yield));
    }

    bool checkDuplicateMessageMistake(const macs::Mid& mid, const macs::BackupMessage& msg, YieldCtx yield) const override {
        return stidByMid(mid, yield) != msg.st_id;
    }

    virtual ~EnvelopeService() = default;
};

} // namespace

FidProviderPtr createFidService(macs::ServicePtr service, macs::Fid defaultFid, std::string folderName, macs::BackupFidsMapping mapping) {
    return std::make_shared<FidService>(service, defaultFid, folderName, mapping);
};


EnvelopeProviderPtr createEnvelopeService(macs::ServicePtr service) {
    return std::make_shared<EnvelopeService>(service);
}


namespace {

class RestoreContext {
private:
    StatePtr state_;
public:
    RestoreContext(StatePtr&& state) {
        state_ = std::move(state);
    }

    void nextState(const Delivery::Message& msg, YieldCtx yield) {
        state_ = state_->next(msg, yield);
    }

    std::optional<Delivery::RestoreResult> getRestoreResult() const {
        return state_->restoreVariables_->result;
    }
};

class InitialState : public Delivery::DeliveryState {
public:
    InitialState(const Delivery& delivery, RestoreVariablesPtr restoreVariables) : DeliveryState(delivery, std::move(restoreVariables)) {}
    StatePtr next(const Delivery::Message& msg, YieldCtx yield) override;

    virtual ~InitialState() = default;
};

class ErrorState : public Delivery::DeliveryState {
public:
    ErrorState(const Delivery& delivery, RestoreVariablesPtr restoreVariables) : DeliveryState(delivery, std::move(restoreVariables)) {}
    StatePtr next(const Delivery::Message& msg, YieldCtx yield) override;

    virtual ~ErrorState() = default;
};

StatePtr InitialState::next(const Delivery::Message& msg, YieldCtx yield) {
    ymod_queuedb::delayOnCancelledTask(getCtx());
    restoreVariables_->nwSmtpCallResult = callNwSmtp(restoreVariables_->fidToStore, msg, yield);
    auto& [renewed, ec] = restoreVariables_->nwSmtpCallResult;

    if (ec) {
        return moveTo<ErrorState>();
    }

    if (restoreVariables_->duplicateMistakeDetected) {
        getEnvelopeProvider()->moveMessage(renewed, msg.tab, restoreVariables_->destinationFid, yield);
    }

    return moveTo<InitialState>(yamail::make_expected(renewed));
}

StatePtr ErrorState::next(const Delivery::Message& msg, YieldCtx yield) {
    auto& [renewed, ec] = restoreVariables_->nwSmtpCallResult;
    auto fidProvider = getFidProvider();
    auto envelopeProvider = getEnvelopeProvider();

    if (ec == DeliveryError::folderIssues) {
        const auto& defaultFid = fidProvider->getDefaultFid();
        if (restoreVariables_->fidToStore != defaultFid) {
            restoreVariables_->fidToStore = defaultFid;
            restoreVariables_->destinationFid = defaultFid;
            return moveTo<InitialState>();
        }
    }
    else if (ec == DeliveryError::duplicateFound) {
        const bool duplicateMessageMistake = envelopeProvider->checkDuplicateMessageMistake(renewed, msg, yield);
        if (!restoreVariables_->duplicateMistakeDetected) {
            if (!duplicateMessageMistake) {
                return moveTo<InitialState>(yamail::make_expected(renewed));
            }
            restoreVariables_->fidToStore = fidProvider->temporaryFid(yield);
            restoreVariables_->duplicateMistakeDetected = true;
            return moveTo<InitialState>();
        }
        if (!duplicateMessageMistake) {
            envelopeProvider->moveMessage(renewed, msg.tab, restoreVariables_->destinationFid, yield);
            return moveTo<InitialState>(yamail::make_expected(renewed));
        }
    }

    return moveTo<ErrorState>(yamail::make_unexpected(ec));
}

} // namespace


struct StoreResponse {
    macs::Mid mid;
};

struct StoreError {
    std::string code;
    std::optional<std::string> message;
};

struct NwSmtpRestoreRequest {
    struct UserInfo {
        std::string email;
    } user_info;

    struct MailInfo {
        std::string stid;
        std::string fid;
        std::string tab;
        std::time_t received_date;
        std::vector<std::string> lids;
        std::vector<std::string> system;
        std::vector<std::string> symbol;
    } mail_info;
};


NwSmtpRestoreRequest storeBody(const std::string& defaultAddress, const macs::Fid& fid, const Delivery::Message& box,
                                const std::vector<std::string>& symbolLabels) {
    return NwSmtpRestoreRequest {
        .user_info {
            .email = defaultAddress
        },
        .mail_info {
            .stid = box.st_id,
            .fid = fid,
            .tab = box.tab.value_or(macs::Tab::Type::relevant).toString(),
            .received_date = box.receivedDate,
            .lids = {},
            .system = {},
            .symbol = symbolLabels
        }
    };
}

template<class Builder>
auto buildRequest(const std::string& defaultAddress, const macs::Fid& fid,
                const ymod_maildb::ServiceParams& params, const Delivery::Message& box,
                const std::vector<macs::Label::Symbol>& symbolLabels, Builder&& post) {
    using namespace http_getter::detail::operators;

    HttpArguments args;

    args.add("uid", params.uid);
    args.add("request_id", params.requestId);
    args.add("service", "barbet");

    http::headers h;
    h.add("Content-Type", "application/json");
    h.add("MIME-Version", "1.0");

    std::vector<std::string> labels(symbolLabels.size());
    std::transform(symbolLabels.begin(), symbolLabels.end(), labels.begin(), [](const macs::Label::Symbol& label) { return label.title(); });

    NwSmtpRestoreRequest body = storeBody(defaultAddress, fid, box, labels);
    std::string text = yamail::data::serialization::JsonWriter<NwSmtpRestoreRequest>(body).result();

    return post.getArgs("args"_arg=args)
        .headers("hdrs"_hdr=h)
        .body(std::move(text));
}

Delivery::MidWithError Delivery::call(const macs::Fid& fid, const Message& msg, YieldCtx yield) const {
    macs::Mid renewed;
    mail_errors::error_code ec = make_error(DeliveryError::temporaryFail, "empty delivery response");
    httpClient->req(
        buildRequest(defaultAddress, fid, serviceParams, msg, symbolLabels, httpClient->toPOST(storeEndpoint))
    )->call("store", [&] (yhttp::response resp) {
        if (resp.status == 200) {
            ec = mail_errors::error_code();
            renewed = yamail::data::deserialization::fromJson<StoreResponse>(resp.body).mid;
            return http_getter::Result::success;
        } else if (resp.status == 406) {
            const auto error = yamail::data::deserialization::fromJson<StoreError>(resp.body);
            const auto& code = error.code;

            if (code == "DuplicateError") {
                renewed = error.message.value();
                ec = make_error(DeliveryError::duplicateFound, fmt::format("(old: {}, new: {})", msg.mid, renewed));
                return http_getter::Result::success;
            } else if (code == "ServiceUnavaliable") {
                ec = make_error(DeliveryError::temporaryFail, std::move(resp.body));
                return http_getter::Result::retry;
            } else if (code == "InvalidFid") {
                ec = make_error(DeliveryError::folderIssues, "invalid fid: "s + fid);
                return http_getter::Result::success;
            } else if (code == "StorageMailNotFound") {
                ec = make_error(DeliveryError::messageNotFound, "absent message, stid: "s + msg.st_id);
                return http_getter::Result::success;
            } else if (code == "NslsPermanentError") {
                ec = make_error(DeliveryError::badMessage, "bad message, stid: "s + msg.st_id);
                return http_getter::Result::success;
            } else {
                ec = make_error(DeliveryError::permanentFail, std::move(resp.body));
                return http_getter::Result::fail;
            }
        } else if (resp.status == 400) {
            ec = make_error(DeliveryError::permanentFail, std::move(resp.body));
            return http_getter::Result::fail;
        } else {
            ec = make_error(DeliveryError::temporaryFail, std::move(resp.body));
            return http_getter::Result::retry;
        }
    }, io_result::make_yield_context(yield));

    return std::make_pair(renewed, ec);
}

inline RestoreContext declareInitialRestoreContext(const Delivery& delivery, const RestoreVariables& restoreVariables) {
    return RestoreContext(
        std::make_unique<InitialState>(delivery, std::make_shared<RestoreVariables>(restoreVariables))
    );
}

yamail::expected<macs::Mid> Delivery::restore(const Message& msg, YieldCtx yield) const {
    macs::Fid destinationFid = fidProvider->resolveFid(msg.fid);
    RestoreVariables contextVariables {destinationFid, destinationFid, false, {}, std::nullopt};
    RestoreContext restoreContext = declareInitialRestoreContext(*this, contextVariables);

    while(!restoreContext.getRestoreResult()) {
        restoreContext.nextState(msg, yield);
    }

    return restoreContext.getRestoreResult().value();
}



std::string DeliveryCategory::message(int v) const {
    switch(DeliveryError(v)) {
        case DeliveryError::permanentFail:
            return "permanent fail";
        case DeliveryError::temporaryFail:
            return "temporary fail";
        case DeliveryError::folderIssues:
            return "folder issues";
        case DeliveryError::duplicateFound:
            return "duplicate found";
        case DeliveryError::messageNotFound:
            return "no data in storage";
        case DeliveryError::badMessage:
            return "bad data in storage";
    }
    return "unknown enum code";
}

const DeliveryCategory& getDeliveryCategory() {
    static DeliveryCategory instance;
    return instance;
}

mail_errors::error_code::base_type make_error_code(DeliveryError e) {
    return mail_errors::error_code::base_type(static_cast<int>(e), getDeliveryCategory());
}

mail_errors::error_code make_error(DeliveryError e, std::string what) {
    return mail_errors::error_code(e, std::move(what));
}

}

YREFLECTION_ADAPT_ADT(barbet::NwSmtpRestoreRequest::UserInfo,
    YREFLECTION_MEMBER(std::string, email)
)

YREFLECTION_ADAPT_ADT(barbet::NwSmtpRestoreRequest::MailInfo,
    YREFLECTION_MEMBER(std::string, stid)
    YREFLECTION_MEMBER(std::string, fid)
    YREFLECTION_MEMBER(std::string, tab)
    YREFLECTION_MEMBER(std::time_t, received_date)
    YREFLECTION_MEMBER(std::vector<std::string>, lids)
    YREFLECTION_MEMBER(std::vector<std::string>, system)
    YREFLECTION_MEMBER(std::vector<std::string>, symbol)
)

YREFLECTION_ADAPT_ADT(barbet::NwSmtpRestoreRequest,
    YREFLECTION_MEMBER(barbet::NwSmtpRestoreRequest::UserInfo, user_info)
    YREFLECTION_MEMBER(barbet::NwSmtpRestoreRequest::MailInfo, mail_info)
)

BOOST_FUSION_ADAPT_STRUCT(barbet::StoreResponse,
    mid
)

BOOST_FUSION_ADAPT_STRUCT(barbet::StoreError,
    code,
    message
)
