#include <mail/barbet/service/include/handlers/backup.h>
#include <mail/barbet/service/include/task_params.h>
#include <mail/barbet/service/include/error.h>
#include <mail/barbet/service/include/setting_guard.h>
#include <mail/barbet/service/include/notify.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>
#include <mail/mail_errors/error_result/error_result.h>

namespace barbet {

bool getUseTabsSettings(macs::ServicePtr service, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    static const std::string tabSetting = "show_folders_tabs";
    macs::settings::ParametersPtr settings = service->settings().getParameters(
        {tabSetting},
        yy
    );

    const auto t = settings->single_settings.find(tabSetting);
    return t != settings->single_settings.end() && t->second == "on";
}

bool backupIsTooFrequent(const macs::BackupStatus& st, std::time_t now, unsigned seconds) {
    std::time_t tt = st.primary ? st.primary->created
                                : std::numeric_limits<std::time_t>::min();

    if (st.additional && st.additional->state != macs::BackupState::error) {
        tt = st.additional->created;
    }

    return tt + seconds >= now;
}

yamail::expected<CreateBackupResult> createBackup(CreateBackupParams params,
                                                  boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    if (const auto e = guardBySetting(params.service, params.guardedBySetting, yy); !e) {
        return yamail::make_unexpected(e.error());
    }

    const macs::BackupStatus st = params.service->backups().getStatus(yy);
    if (backupIsTooFrequent(st, std::time(nullptr), params.secondsBetweenBackups)) {
        return make_unexpected(ServiceError::frequentBackup);
    }

    const macs::BackupId id = params.service->backups().backupId(yy);
    const ymod_queuedb::RequestId requestId(params.macsParams.sr.requestId);

    const auto taskId = params.queuedb.addTask(
        params.uid, fillBackupType(),
        dumpFillBackupParams(id, params.macsParams), params.timeout, requestId,
        yy
    );

    LOGDOG_(params.logger, notice,
            log::message="task created",
            log::task_type=fillBackupType(),
            log::task_id=taskId);

    const bool useTabs = getUseTabsSettings(params.service, yield);

    params.service->backups().createBackup(
        id, params.maxMessages, useTabs,
        yy
    );

    return CreateBackupResult {
        .backup_id=id
    };
}

yamail::expected<void> fillBackupImpl(FillBackupParams& params, const ExecOrWait& execOrWait,
                                      yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    bool selectedBackupIsInProgress = false;
    execOrWait([&]() {
        const macs::BackupStatus st = params.service->backups().getStatus(yy);

        if (st.primary && st.primary->backup_id == params.backupId) {
            return true;
        }

        if (st.additional && st.additional->backup_id == params.backupId) {
            selectedBackupIsInProgress = true;
            return true;
        }

        return false;
    });

    if (!selectedBackupIsInProgress) {
        return yamail::make_expected();
    }
    ymod_queuedb::delayOnCancelledTask(ctx);

    const bool useTabs = getUseTabsSettings(params.service, yield);

    ymod_queuedb::delayOnCancelledTask(ctx);

    params.service->backups().fillBackup(params.backupId, useTabs, yy);
    return yamail::make_expected();
}

yamail::expected<void> fillBackup(FillBackupParams params, const ExecOrWait& eow, bool lastTry,
                                  yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    yamail::expected<void> result;
    try {
        result = fillBackupImpl(params, eow, ctx, yield);
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const boost::system::system_error& e) {
        result = yamail::make_unexpected(mail_errors::error_code(e.code()));
    } catch (const std::exception& e) {
        result = make_unexpected(ServiceError::unexpectedException, e.what());
    }

    if (!result && lastTry) {
        const auto notified = notifyCreateFailed(params, yield);
        if (!notified) {
            return ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay);
        }

        mail_errors::error_code ec;
        params.service->backups().failRunningBackup(
            mail_errors::makeErrorResult(result.error()),
            io_result::make_yield_context(yield, ec)
        );
        if (ec) {
            LOGDOG_(params.logger, error,
                    log::message="cannot fail running backup",
                    log::error_code=ec);
            return ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay);
        }
    }

    return result;
}

yamail::expected<DeactivateBackupResult> deactivateBackup(DeactivateBackupParams params,
                                                          boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    params.service->backups().deactivateBackup(yy);

    return DeactivateBackupResult();
}

}
