#include <macs/folders_repository.h>
#include <mail/barbet/service/include/handlers/restore.h>
#include <mail/barbet/service/include/task_params.h>
#include <mail/barbet/service/include/helpers.h>
#include <mail/barbet/service/include/error.h>
#include <mail/barbet/service/include/blackbox.h>
#include <mail/barbet/service/include/delivery.h>
#include <mail/barbet/service/include/setting_guard.h>
#include <mail/barbet/service/include/notify.h>
#include <mail/ymod_queuedb_worker/include/task_control.h>
#include <mail/mail_errors/error_result/error_result.h>


namespace barbet {

macs::BackupFidsMapping restoredFolder(macs::BackupFolders backuped,
                                       macs::Fid special) {

    macs::BackupFidsMapping ret;
    for (const macs::BackupFolder& folder: backuped) {
        if (folder.isBackuped) {
            ret[folder.fid] = special;
        }
    }

    return ret;
}

void checkErrorCode(const mail_errors::error_code& ec) {
    if (ec.category() == macs::error::getCategory() && (
            ec == macs::error::noSuchFolder       || ec == macs::error::folderAlreadyExists ||
            ec == macs::error::folderCantBeParent || ec == macs::error::foldersLimitExceeded
        )
    ) {
        throw mail_errors::system_error(
            make_error(ServiceError::folderIssues, ec.full_message())
        );
    }
}

macs::BackupFidsMapping fullHierarchy(macs::BackupFolders backuped,
                                      const macs::FoldersRepository& repo,
                                      boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    const auto paths = detail::makeBackupFoldersPaths(backuped);
    macs::BackupFidsMapping ret;
    const macs::FolderSet folders = repo.getAllFolders(yy);
    for (const macs::BackupFolder& bFolder: backuped) {
        if (bFolder.isBackuped) {
            macs::Folder::Path path(bFolder.name);
            if (auto p = paths.find(bFolder.fid); p != paths.end()) {
                path = p->second;
            }
            const auto it = folders.find(path);

            if (it == folders.end()) {
                mail_errors::error_code ec;
                macs::Fid fid = repo.createFolderByPath(path, yy[ec]).fid();

                checkErrorCode(ec);

                ret[bFolder.fid] = fid;
            } else {
                ret[bFolder.fid] = it->second.fid();
            }
        }
    }

    const macs::Fid inbox = folders.fid(macs::Folder::Symbol::inbox);
    ret[inbox] = inbox;

    return ret;
}

yamail::expected<RestoreResult> restore(RestoreParams params, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    if (const auto e = guardBySetting(params.service, params.guardedBySetting, yy); !e) {
        return yamail::make_unexpected(e.error());
    }

    const macs::BackupStatus st = params.service->backups().getStatus(yy);

    if (!st.primary) {
        return make_unexpected(ServiceError::backupInWrongState, "missing 'complete' backup");
    }

    const macs::BackupId& id = st.primary->backup_id;

    macs::BackupFidsMapping mapping;
    macs::BackupFolders backuped = params.service->backups().folders(id, yy);
    macs::Fid restored = macs::getOrCreateFolderBySymbolWithRandomizedName(params.service->folders(),
        "Restored", macs::Folder::noParent, macs::Folder::Symbol::restored, false, yy
    ).fid();
    if (params.kind == RestoreParams::Kind::restored_folder) {
        mapping = restoredFolder(std::move(backuped), std::move(restored));
    } else {
        mapping = fullHierarchy(std::move(backuped), params.service->folders(), yield);
    }

    const std::time_t now = std::time(nullptr);
    const ymod_queuedb::RequestId requestId(params.macsParams.sr.requestId);
    const auto taskId = params.queuedb.addTask(
        params.uid, fillRestoreType(),
        dumpFillRestoreParams(id, now, params.macsParams), params.timeout, requestId,
        yy
    );

    LOGDOG_(params.logger, notice,
            log::message="task created",
            log::task_type=fillRestoreType(),
            log::task_id=taskId);

    params.service->backups().restore(id, now, params.kind, mapping, yy);

    return RestoreResult();
}

template<class YY>
void moveFromFolder(const macs::Fid& fid, const macs::BackupsRepository& backups,
                    const macs::Fid& defaultFid, const macs::BackupFidsMapping& mapping,
                    const FillRestoreParams& params, yplatform::task_context_ptr ctx, YY& yy) {
    do {
        const macs::BackupBox messages = backups.selectMessagesInFolder(
            params.backupId, params.hiddenMessagesChunk, fid,
            yy
        );

        if (messages.empty()) {
            break;
        }
        ymod_queuedb::delayOnCancelledTask(ctx);

        using TabOpt = std::optional<macs::Tab::Type>;
        std::map<std::tuple<macs::Fid, TabOpt>, std::list<macs::Mid>> groupedMids;
        for (const auto& m : messages) {
            macs::Fid fid = helpers::resolveFid(m.fid, defaultFid, mapping);
            groupedMids[std::make_tuple(fid, m.tab)].emplace_back(m.mid);
        }

        for (const auto& [fidTab, mids] : groupedMids) {
            params.service->envelopes().moveMessages(
                std::get<macs::Fid>(fidTab), std::get<TabOpt>(fidTab), mids,
                yy
            );
        }
    } while (true);
}

yamail::expected<void> fillRestoreImpl(FillRestoreParams& params, const ExecOrWait& execOrWait,
                                       yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);

    const macs::BackupsRepository& backups = params.service->backups();

    std::optional<macs::Restore> restore;
    execOrWait([&] () {
        macs::BackupStatus st = backups.getStatus(yy);

        if (st.restore && st.restore->backup_id == params.backupId && st.restore->created == params.now) {
            restore = std::move(st.restore);
            return true;
        }

        return false;
    });

    if (!restore) {
        return yamail::make_expected();
    }
    ymod_queuedb::delayOnCancelledTask(ctx);

    const macs::FolderSet folders = params.service->folders().getAllFoldersWithHidden(yy);

    const macs::Fid defaultFid = folders.fid(macs::Folder::Symbol::restored);
    const macs::Fid trashFid = folders.fid(macs::Folder::Symbol::trash);
    const macs::Fid hiddenTrashFid = folders.fid(macs::Folder::Symbol::hidden_trash);

    if (!hiddenTrashFid.empty() && folders.at(hiddenTrashFid).messagesCount() > 0) {
        moveFromFolder(hiddenTrashFid, backups, defaultFid, restore->fids_mapping, params, ctx, yy);
    }

    if (!trashFid.empty() && folders.at(trashFid).messagesCount() > 0) {
        moveFromFolder(trashFid, backups, defaultFid, restore->fids_mapping, params, ctx, yy);
    }

    const auto address = defaultAddress(params.uid, *params.client, params.blackbox, yield);
    if (!address) {
        return yamail::make_unexpected(address.error());
    }
    ymod_queuedb::delayOnCancelledTask(ctx);

    auto fidServicePtr = createFidService(params.service, defaultFid, std::to_string(params.backupId), restore->fids_mapping);
    auto envelopeServicePtr = createEnvelopeService(params.service);
    Delivery delivery(address.value(), {}, fidServicePtr, envelopeServicePtr, params, ctx);
    const macs::EnvelopesRepository& envelopes = params.service->envelopes();

    macs::BackupBox messages;
    do {
        messages = backups.selectMessages(params.backupId, params.messagesChunk, yy);
        ymod_queuedb::delayOnCancelledTask(ctx);

        macs::BackupMidToMid midToMid;
        macs::MidVec midsToDelete;
        for (auto& msg : messages) {
            if (helpers::isMulcaStid(msg.st_id)) {
                LOGDOG_(params.logger, notice,
                    log::message=fmt::format("ignore mulca-only stid: '{}'", msg.st_id),
                    log::task_id=params.taskId
                );
                midsToDelete.emplace_back(std::move(msg.mid));
                continue;
            }

            const auto renewed = delivery.restore(msg, yield);
            if (!renewed) {
                return renewed.get_unexpected();
            }

            auto restoredMid = renewed.value();
            midToMid[msg.mid] = restoredMid;
            
            envelopes.updateMailAttributes(restoredMid, msg.attributes, yy);
        }

        if (!midsToDelete.empty()) {
            backups.deleteMessages(params.backupId, midsToDelete, yy);
        }

        if (!midToMid.empty()) {
            backups.updateMessages(params.backupId, midToMid, yy);
        }
        
    } while (!messages.empty());

    fidServicePtr->clearTemporaryFids(yield);
    backups.completeRunningRestore(yy);

    return yamail::make_expected();
}

yamail::expected<void> fillRestore(FillRestoreParams params, const ExecOrWait& eow, bool lastTry,
                                   yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    yamail::expected<void> result;
    try {
        result = fillRestoreImpl(params, eow, ctx, yield);
    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const boost::system::system_error& e) {
        result = yamail::make_unexpected(mail_errors::error_code(e.code()));
    } catch (const std::exception& e) {
        result = make_unexpected(ServiceError::unexpectedException, e.what());
    }

    if (!result && (lastTry || result.error() != DeliveryError::temporaryFail)) {
        const auto notified = notifyRestoreFailed(params, yield);
        if (!notified) {
            return lastTry ? ymod_queuedb::make_unexpected(ymod_queuedb::TaskControl::delay)
                                  : notified;
        }

        mail_errors::error_code ec;
        params.service->backups().failRunningRestore(
            mail_errors::makeErrorResult(result.error()),
            io_result::make_yield_context(yield, ec)
        );
        if (ec) {
            LOGDOG_(params.logger, error,
                    log::message="cannot fail running restore",
                    log::error_code=ec);
            return yamail::make_unexpected(lastTry ? make_error(ymod_queuedb::TaskControl::delay) : ec);
        }
    }
    return result;
}

namespace detail {
std::map<macs::Fid, macs::Folder::Path> makeBackupFoldersPaths(const macs::BackupFolders& folders) {
    std::map<macs::Fid, macs::BackupFolder> folder_map;
    for (auto& f: folders) {
        folder_map.emplace(f.fid, f);
    }
    std::map<macs::Fid, macs::Folder::Path> ret;
    for (auto& f: folders) {
        std::vector<macs::Folder::Name> revertPath = {f.name};
        macs::Fid parentFid = f.parentFid.value_or("");
        while (parentFid != "" && parentFid != "0") {
            if (auto parent = folder_map.find(parentFid); parent != folder_map.end()) {
                revertPath.push_back(parent->second.name);
                parentFid = parent->second.parentFid.value_or("");
            } else {
                break;
            }
        }
        ret.emplace(f.fid, macs::Folder::Path(revertPath.rbegin(), revertPath.rend()));
    }
    return ret;
}
}
}
