#include <mail/barbet/service/include/handlers/settings.h>
#include <mail/webmail/http_api_helpers/include/error.h>

namespace barbet {

bool findUnbackupableFolder(const macs::FidVec& fids, const macs::FolderSet& folders) {
    const macs::Folder& spam = folders.at(macs::Folder::Symbol::spam);
    const macs::Folder& trash = folders.at(macs::Folder::Symbol::trash);
    const macs::Fid& hiddenTrashFid = folders.fid(macs::Folder::Symbol::hidden_trash);

    return !std::all_of(fids.begin(), fids.end(), [&] (const macs::Fid& fid) {
        return fid != spam.fid()
            && fid != trash.fid()
            && (hiddenTrashFid == macs::Fid() || fid != hiddenTrashFid);
    });
}

yamail::expected<SettingsResult> settings(SettingsParams params, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    return SettingsResult(params.service->backups().getSettings(yy));
}

yamail::expected<SettingsResult> updateSettings(UpdateSettingsParams params, boost::asio::yield_context yield) {
    std::vector<macs::Tab::Type> tabs;
    tabs.reserve(params.tabs.size());
    for (auto& element: params.tabs) {
        auto tab = macs::Tab::Type::fromString(element, std::nothrow);
        if (tab == macs::Tab::Type::unknown) {
            return make_unexpected(http_api::Error::invalidArgument, "tab '" + element + "' unknown");
        }
        tabs.push_back(tab);
    }

    auto isEmptyFid = [] (const macs::Fid& f) { return f.empty(); };
    if (std::any_of(params.fids.begin(), params.fids.end(), isEmptyFid)) {
        return make_unexpected(http_api::Error::invalidArgument, "fids contains empty element");
    }

    const auto yy = io_result::make_yield_context(yield);
    const macs::FolderSet folders = params.service->folders().getAllFoldersWithHidden(yy);

    if (findUnbackupableFolder(params.fids, folders)) {
        return make_unexpected(http_api::Error::invalidArgument,
                               "fids contains unbackupable folder (spam, trash, hidden_trash)");
    }

    return SettingsResult(params.service->backups().changeSettings(
        macs::BackupSettings { .fids=std::move(params.fids), .tabs=std::move(tabs) },
        yy
    ));
}

}
