#include <mail/barbet/service/include/hound.h>
#include <mail/barbet/service/include/error.h>

#include <yamail/data/deserialization/json_reader.h>


namespace barbet::hound {

struct HoundError {
    std::string reason;
};

struct HoundResponse {
    std::optional<HoundError> error;
};


yamail::expected<void> activateUserAndChangeArchiveState(const std::string& uid, const std::string_view action, const http_getter::Client& client, 
                                                                const http_getter::Endpoint& houndEndpoint, boost::asio::yield_context yield) {
    auto endpoint = houndEndpoint.format(fmt::arg("uid", uid));
    auto requestBuilder = client.toPOST(endpoint).body(yhttp::form_encode({ {"action", action.data()} }));

    yamail::expected<void> result = make_unexpected(ServiceError::unexpectedCondition, "can't get response from hound");
    client.req(requestBuilder)
    ->call("hound_archive", [&result](yhttp::response resp) -> http_getter::Result {
        if (http_getter::helpers::successCode(resp.status)) {
            result = {};
            return http_getter::Result::success;
        }

        std::optional<std::string> reason;
        try {
            auto houndResp = yamail::data::deserialization::fromJson<HoundResponse>(resp.body);
            if (houndResp.error) {
                reason = std::move(houndResp.error->reason);
            }
        } catch(const std::exception& e) {
            reason = e.what();
        } catch(...) {
            reason = "unexpected exception type";
        }
         
        if (http_getter::helpers::retriableCode(resp.status)) {
            result = make_unexpected(ServiceError::retriableError, reason.value_or("try later"));
            return http_getter::Result::retry;
        }

        result = make_unexpected(ServiceError::nonRetriableError, reason.value_or("unknown error"));
        return http_getter::Result::fail;
    }
    , io_result::make_yield_context(yield));

    return result;
}

}


YREFLECTION_ADAPT_ADT(barbet::hound::HoundError,
    YREFLECTION_MEMBER(std::string, reason)
)
 
YREFLECTION_ADAPT_ADT(barbet::hound::HoundResponse,
    YREFLECTION_MEMBER(std::optional<barbet::hound::HoundError>, error)
)
