#pragma once

#include <mail/barbet/service/include/params.h>
#include <yamail/expected.h>

namespace barbet {

using YieldCtx = boost::asio::yield_context;


class FidProvider {
public:
    virtual macs::Fid temporaryFid(YieldCtx yield) = 0;
    virtual const macs::Fid& getDefaultFid() const = 0;
    virtual macs::Fid resolveFid(const macs::Fid& fid) const = 0;
    virtual void clearTemporaryFids(YieldCtx yield) = 0;

    virtual ~FidProvider() = default;
};


class EnvelopeProvider {
public:
    virtual macs::Stid stidByMid(const macs::Mid& mid, YieldCtx yield) const = 0;
    virtual void moveMessage(const macs::Mid& mid, const std::optional<macs::Tab::Type>& tab, const macs::Fid& fid, YieldCtx yield) const = 0;
    virtual bool checkDuplicateMessageMistake(const macs::Mid& mid, const macs::BackupMessage& msg, YieldCtx yield) const = 0;

    virtual ~EnvelopeProvider() = default;
};


using FidProviderPtr = std::shared_ptr<FidProvider>;
using EnvelopeProviderPtr = std::shared_ptr<EnvelopeProvider>;


FidProviderPtr createFidService(macs::ServicePtr service, macs::Fid defaultFid, std::string folderName, macs::BackupFidsMapping mapping);
EnvelopeProviderPtr createEnvelopeService(macs::ServicePtr service);


class Delivery {
public:
    using Message = macs::BackupMessage;
    using SymbolLabels = std::vector<macs::Label::Symbol>;
    using RestoreResult = yamail::expected<macs::Mid>;
    using MidWithError = std::pair<macs::Mid, mail_errors::error_code>;
private:
    std::string defaultAddress;
    SymbolLabels symbolLabels;
    yplatform::task_context_ptr ctx;
    FidProviderPtr fidProvider;
    EnvelopeProviderPtr envelopeProvider;
    ymod_maildb::ServiceParams serviceParams;
    http_getter::ClientPtr httpClient;
    http_getter::Endpoint storeEndpoint;
protected:
    MidWithError call(const macs::Fid& fid, const Message& msg, YieldCtx yield) const;

public:
    virtual ~Delivery() = default;

    Delivery(std::string defaultAddress, SymbolLabels symbolLabels,
            std::shared_ptr<FidProvider> fidProvider,
            std::shared_ptr<EnvelopeProvider> envelopeProvider,
            const FillRestoreParams& params, yplatform::task_context_ptr ctx)
        : defaultAddress(std::move(defaultAddress))
        , symbolLabels(std::move(symbolLabels))
        , ctx(std::move(ctx))
        , fidProvider(std::move(fidProvider))
        , envelopeProvider(std::move(envelopeProvider))
        , serviceParams(params.macsParams.sr)
        , httpClient(params.client)
        , storeEndpoint(params.store)
        { }

    Delivery(std::string defaultAddress_, SymbolLabels symbolLabels_,
            std::shared_ptr<FidProvider> fidProvider_,
            std::shared_ptr<EnvelopeProvider> envelopeProvider_,
            ymod_maildb::ServiceParams serviceParams_, http_getter::ClientPtr httpClient_,
            http_getter::Endpoint storeEndpoint_, yplatform::task_context_ptr ctx_)
        : defaultAddress(std::move(defaultAddress_))
        , symbolLabels(std::move(symbolLabels_))
        , ctx(std::move(ctx_))
        , fidProvider(std::move(fidProvider_))
        , envelopeProvider(std::move(envelopeProvider_))
        , serviceParams(std::move(serviceParams_))
        , httpClient(std::move(httpClient_))
        , storeEndpoint(std::move(storeEndpoint_))
        { }

    RestoreResult restore(const Message& msg, YieldCtx) const;

    class DeliveryState;
};

using StatePtr = std::unique_ptr<Delivery::DeliveryState>;

struct RestoreVariables {
    macs::Fid destinationFid;
    macs::Fid fidToStore;
    bool duplicateMistakeDetected;
    Delivery::MidWithError nwSmtpCallResult;
    std::optional<Delivery::RestoreResult> result;
};

using RestoreVariablesPtr = std::shared_ptr<RestoreVariables>;

class Delivery::DeliveryState {
public:
    const Delivery& delivery_;
    RestoreVariablesPtr restoreVariables_;

    DeliveryState(const Delivery& delivery, RestoreVariablesPtr restoreVariables) 
        : delivery_(delivery)
        , restoreVariables_(std::move(restoreVariables))
        { }

    virtual ~DeliveryState() = default;

    virtual StatePtr next(const Delivery::Message& msg, YieldCtx yield) = 0;

    template<class State>
    StatePtr moveTo() const {
        return std::make_unique<State>(delivery_, restoreVariables_);
    }

    template<class State>
    StatePtr moveTo(Delivery::RestoreResult&& result) {
        restoreVariables_->result = std::move(result);
        return moveTo<State>();
    }

    yplatform::task_context_ptr getCtx() const { return delivery_.ctx; }
    FidProviderPtr getFidProvider() const { return delivery_.fidProvider; }
    EnvelopeProviderPtr getEnvelopeProvider() const { return delivery_.envelopeProvider; }

    Delivery::MidWithError callNwSmtp(const macs::Fid& fidToStore, const Delivery::Message& msg, YieldCtx yield) const { 
        return delivery_.call(fidToStore, msg, yield); 
    }
};

enum class DeliveryError {
    permanentFail = 1,
    temporaryFail,
    folderIssues,
    duplicateFound,
    messageNotFound,
    badMessage,
};

struct DeliveryCategory : public mail_errors::error_code::error_category {
    const char* name() const noexcept override {
        return "delivery";
    }

    std::string message(int v) const override;
};
const DeliveryCategory& getDeliveryCategory();
mail_errors::error_code::base_type make_error_code(DeliveryError e);
mail_errors::error_code make_error(DeliveryError e, std::string what = "");

} //namespace barbet

namespace boost::system {
template <>
struct is_error_code_enum<barbet::DeliveryError> : std::true_type {};
}
