#include <mail/barbet/service/include/s3_methods.h>
#include <mail/barbet/service/include/error.h>
#include <spdlog/details/format.h>
#include <yamail/data/deserialization/json_reader.h>

#include <aws/s3/model/GetObjectRequest.h>
#include <aws/s3/model/ListObjectsV2Request.h>

#include <boost/algorithm/string.hpp>

BOOST_FUSION_ADAPT_STRUCT(barbet::archive::ArchiveMessage,
    received_date,
    st_id,
    folder_type,
    is_shared
)

namespace barbet::archive {

namespace {

inline auto wrapS3Error(const auto& outcome) {
    const auto& e = outcome.GetError();
    return make_error(ServiceError::unexpectedException,
                fmt::format("{}: {}", e.GetExceptionName(), e.GetMessage())
    );
}

inline ArchiveChunk chunkParser(const Aws::S3::Model::Object& o) {
    constexpr std::string_view CHUNK_KEY_SEPARATORS = "/_";
    constexpr size_t PARTS_COUNT = 3;
    auto& key = o.GetKey();
    std::vector<std::string> parts;
    parts.reserve(PARTS_COUNT);
    boost::split(parts, key, boost::is_any_of(CHUNK_KEY_SEPARATORS));
    if (PARTS_COUNT != parts.size()) {
        throw std::invalid_argument(fmt::format("invalid key format, key={}", key));
    }
    
    return ArchiveChunk{ 
        .key = key,
        .last_mid = boost::lexical_cast<int64_t>(parts[1]),
        .count = boost::lexical_cast<int64_t>(parts[2]),
    };
}

}

std::vector<ArchiveChunk> getUserArchiveChunks(const std::string& uid, const ymod_s3::ClientPtr& s3,
                                                const std::string& s3Bucket, boost::asio::yield_context yield) {
    constexpr char FOLDER_SEPARATOR = '/';
    std::vector<ArchiveChunk> res;

    const auto yy = io_result::make_yield_context(yield);
    auto listObectsReq = Aws::S3::Model::ListObjectsV2Request()
                        .WithBucket(s3Bucket)
                        .WithPrefix(fmt::format("{}{}", uid, FOLDER_SEPARATOR))
                        .WithMaxKeys(1000);

    bool isTruncated = false;
    do {
        auto objectsOutcome = s3->ListObjectsV2(listObectsReq, yy);
        if (!objectsOutcome.IsSuccess()) {
            throw mail_errors::system_error(wrapS3Error(objectsOutcome));
        }

        auto& objectsResult = objectsOutcome.GetResult();
        isTruncated = objectsResult.GetIsTruncated();
        listObectsReq.SetContinuationToken(objectsResult.GetNextContinuationToken());

        auto& objects = objectsResult.GetContents();
        std::transform(objects.begin(), objects.end(), std::back_inserter(res), chunkParser);

    } while(isTruncated);

    return res;
}

std::vector<ArchiveMessage> getArchiveMessages(const std::string& key, const ymod_s3::ClientPtr& s3,
                                                const std::string& s3Bucket, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    auto getObjectReq = Aws::S3::Model::GetObjectRequest()
                        .WithBucket(s3Bucket)
                        .WithKey(key);
    auto objectOutcome = s3->GetObject(getObjectReq, yy);
    if (!objectOutcome.IsSuccess()) {
        throw mail_errors::system_error(wrapS3Error(objectOutcome));
    }

    std::stringstream ss;
    ss << objectOutcome.GetResult().GetBody().rdbuf();

    return yamail::data::deserialization::fromJson<std::vector<ArchiveMessage>>(ss.str());
}

}
