#include <mail/barbet/service/include/tasks/archive.h>
#include <mail/barbet/service/include/blackbox.h>
#include <mail/barbet/service/include/delivery.h>
#include <mail/barbet/service/include/helpers.h>

#include <mail/barbet/service/include/unarchiver.h>
#include <mail/barbet/service/include/s3_methods.h>

#include <mail/ymod_queuedb_worker/include/task_control.h>
#include <mail/mail_errors/error_result/error_result.h>

#include <util/generic/scope.h>


namespace barbet::archive {

namespace {

class DeliveryGuy : public Postman {
public:
    explicit DeliveryGuy(macs::ServicePtr service_, Delivery&& delivery_, ContextLogger& logger_, ymod_queuedb::TaskId taskId_)
        : service{std::move(service_)}
        , delivery{std::move(delivery_)}
        , logger{logger_}
        , taskId{taskId_}
        { }

    Mid restore(const Message& postmanMsg, boost::asio::yield_context yield) const override {
        if (helpers::isMulcaStid(postmanMsg.st_id)) {
            LOGDOG_(logger, notice,
                log::message=fmt::format("ignore mulca-only stid: '{}'", postmanMsg.st_id),
                log::task_id=taskId
            );
            return {};
        }
        Delivery::Message msg = postmanToDeliveryMsg(postmanMsg);
        auto restored = delivery.restore(msg, yield);
        if (!restored) {
            if (restored.error() == DeliveryError::messageNotFound) {
                LOGDOG_(logger, notice, log::message=fmt::format("ignore absent stid: '{}'", postmanMsg.st_id), log::task_id=taskId);
                return {};
            } else if (restored.error() == DeliveryError::badMessage) {
                LOGDOG_(logger, notice, log::message=fmt::format("ignore bad message, stid: '{}'", postmanMsg.st_id), log::task_id=taskId);
                return {};
            } else {
                throw mail_errors::system_error(restored.error());
            }
        }
        Mid restoredMid = restored.value();

        const auto yy = io_result::make_yield_context(yield);
        service->envelopes().updateMailAttributes(restoredMid, msg.attributes, yy);

        return restoredMid;
    }

private:
    macs::ServicePtr service;
    Delivery delivery;
    ContextLogger& logger;
    ymod_queuedb::TaskId taskId;

    static Delivery::Message postmanToDeliveryMsg(const Postman::Message& msg) {
        std::vector<std::string> attributes;
        if (msg.is_shared) {
            attributes = {"mulca-shared"};
        }
        return Delivery::Message {
            .st_id = msg.st_id,
            .receivedDate = msg.received_date,
            .attributes = std::move(attributes)
        };
    }
};


class S3Unarchiver : public Unarchiver {
public:
    S3Unarchiver(ymod_s3::ClientPtr s3_, std::string s3Bucket_, macs::ServicePtr service_,
        unsigned int updateInterval_, macs::TaskId taskId_, yplatform::task_context_ptr ctx_ )
        : s3{std::move(s3_)}
        , s3Bucket{std::move(s3Bucket_)}
        , service{std::move(service_)}
        , updateInterval{updateInterval_}
        , taskId{taskId_}
        , ctx{std::move(ctx_)}
        { }
protected:
    std::vector<ArchiveChunk> getUserArchiveChunks(const std::string& uid, YieldCtx yield) const override {
        return archive::getUserArchiveChunks(uid, s3, s3Bucket, std::move(yield));
    }

    std::vector<ArchiveMessage> getArchiveMessages(const std::string& key, YieldCtx yield) const override {
        return archive::getArchiveMessages(key, s3, s3Bucket, std::move(yield));
    }

    void restorationProgress(std::int64_t currentlyRestored, YieldCtx yield) const override {
        if (needProgressUpdate(currentlyRestored)) {
            updateRestorationProgress(currentlyRestored, std::move(yield));
        }
    }

    void restorationError(std::int64_t currentlyRestored, YieldCtx yield) const override {
        updateRestorationProgress(currentlyRestored, std::move(yield));
    }

    void restorationComplete(std::int64_t totallyRestored, YieldCtx yield) const override {
        if (!needProgressUpdate(totallyRestored)) {
            updateRestorationProgress(totallyRestored, std::move(yield));
        }
    }

    void updateRestorationProgress(std::int64_t currentlyRestored, YieldCtx yield) const {
        const auto yy = io_result::make_yield_context(yield);
        service->users().updateRestorationProgress(taskId, currentlyRestored, yy);
        ymod_queuedb::delayOnCancelledTask(ctx);
    }

    inline bool needProgressUpdate(std::int64_t currentlyRestored) const {
        return ctx->is_cancelled() || (0 == (currentlyRestored % updateInterval));
    }


    ymod_s3::ClientPtr s3;
    std::string s3Bucket;

    macs::ServicePtr service;
    unsigned int updateInterval;

    macs::TaskId taskId;
    yplatform::task_context_ptr ctx;
};


bool archiveRestoreAcquireLock(const archive::RestoreTaskParams& params, const ExecOrWait& execOrWait,
                                const yplatform::task_context_ptr& ctx, boost::asio::yield_context yield) {
    using namespace macs;
    constexpr auto archiveState = ArchiveState::restoration_in_progress;
    const auto& userRepo = params.service->users();

    const auto yy = io_result::make_yield_context(yield);
    const auto taskId = params.taskId;
    const auto getArchive = [&]() -> Archive {
        std::optional<Archive> arch = userRepo.getArchive(yy);
        if (!arch) {
            throw std::runtime_error("archive not found");
        }
        return std::move(arch.value());
    };

    Archive archive = getArchive();
    const auto isPerformer = [&]() {
        ymod_queuedb::delayOnCancelledTask(ctx);
        return (archiveState == archive.state) && (taskId == ymod_queuedb::TaskId{archive.task_id.value_or(EMPTY_TASK_ID)});
    };

    if (isPerformer()) {
        return true;
    }

    execOrWait([&] () {
        if (archiveState == archive.state) {
            userRepo.exchangeArchiveTaskId(EMPTY_TASK_ID, taskId, yy);
        }

        archive = getArchive();
        return isPerformer();
    });

    if (ArchiveState::restoration_requested == archive.state) {
        LOGDOG_(params.logger, warning,
            log::message="user's archive still in 'restoration_requested' state",
            log::task_id=taskId
        );
        throw mail_errors::system_error(make_error(ymod_queuedb::TaskControl::delay));
    }

    return isPerformer();
}

std::pair<Delivery, std::shared_ptr<FidProvider>> createDelivery(const archive::RestoreTaskParams& params,
                                    yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    const auto yy = io_result::make_yield_context(yield);
    const macs::ServicePtr& service = params.service;
    const macs::Fid defaultFid = macs::getOrCreateFolderBySymbolWithRandomizedName(service->folders(),
        "Restored", macs::Folder::noParent, macs::Folder::Symbol::restored, false, yy
    ).fid();

    const auto address = defaultAddress(params.uid, *params.client, params.blackbox, yield);
    if (!address) {
        throw mail_errors::system_error(address.error());
    }

    const auto& serviceParams = params.macsParams.sr;
    auto fidServicePtr = createFidService(service, defaultFid, serviceParams.uid, macs::BackupFidsMapping{});
    auto envelopeServicePtr = createEnvelopeService(service);
    auto delivery = Delivery{address.value(), {macs::Label::Symbol::seen_label}, fidServicePtr, envelopeServicePtr, serviceParams,
                      params.client, params.store, std::move(ctx)};

    return std::make_pair(std::move(delivery), std::move(fidServicePtr));
}

S3Unarchiver createUnarchiver(const archive::RestoreTaskParams& params, yplatform::task_context_ptr ctx) {
    return S3Unarchiver{params.s3, params.s3Bucket, params.service, params.updateInterval, params.taskId, std::move(ctx)};
}

bool isDeliveryNonTemporaryFail(const yamail::expected<void>& res) {
    return !res && res.error().category() == getDeliveryCategory() && res.error() != DeliveryError::temporaryFail;
}


}

yamail::expected<void> archiveRestore(archive::RestoreTaskParams params, const ExecOrWait& eow, bool lastTry,
                                   yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
    using namespace macs;
    constexpr auto archiveState = ArchiveState::restoration_in_progress;
    yamail::expected<void> result;
    const auto taskId = params.taskId;
    const auto yy = io_result::make_yield_context(yield);
    const auto& userRepo = params.service->users();

    try {
        if (!archiveRestoreAcquireLock(params, eow, ctx, yield)) {
            return result;
        }
        Y_SCOPE_EXIT(&userRepo, taskId, yy) {
            userRepo.exchangeArchiveTaskId(taskId, EMPTY_TASK_ID, yy);
        };

        auto [delivery, fidServicePtr] = createDelivery(params, ctx, yield);
        DeliveryGuy postman{params.service, std::move(delivery), params.logger, taskId};
        auto unarchiver = createUnarchiver(params, ctx);
        std::int64_t previouslyRestored = userRepo.getArchive(yy)->restored_message_count;

        LOGDOG_(params.logger, notice,
            log::message="start working on the user's archive",
            log::task_id=taskId
        );

        unarchiver.restore(params.uid, previouslyRestored, postman, yield);
        fidServicePtr->clearTemporaryFids(yield);

        userRepo.exchangeArchiveState(
            UserState::active,
            archiveState, ArchiveState::restoration_complete,
            std::nullopt, yy
        );

    } catch (const boost::coroutines::detail::forced_unwind&) {
        throw;
    } catch (const boost::system::system_error& e) {
        result = yamail::make_unexpected(mail_errors::error_code(e.code(), e.what()));
    } catch (const std::exception& e) {
        result = make_unexpected(ServiceError::unexpectedException, e.what());
    }

    if (!result && (lastTry || isDeliveryNonTemporaryFail(result))) {
        mail_errors::error_code ec;

        using namespace macs;
        userRepo.exchangeArchiveState(
            UserState::active,
            archiveState, ArchiveState::restoration_error,
            mail_errors::makeErrorResult(result.error()), io_result::make_yield_context(yield, ec)
        );

        if (ec) {
            LOGDOG_(params.logger, error,
                    log::message="cannot fail archive restoration",
                    log::error_code=ec);

            return yamail::make_unexpected(lastTry ? make_error(ymod_queuedb::TaskControl::delay) : ec);
        }
    }
    return result;
}

}
